;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_ADD_CD
;
; Purpose     :	Adds CDROM DATA files to current file selection.
;
; Explanation : Adds CDROM DATA files to current file selection.
;               This routine uses the CD volume labels to determine which days are present.
;               If days present between TSTART and TSTOP then checks files exist and adds them to
;               respective NM, BM or HK file streams. If files do not exist them issues warning.
;
; Use         : < days = rcs_add_cd(ids) >
;
; Inputs      : ids : STRUCTURE containing widget IDs.
;
; Opt. Inputs : None.
;
; Outputs     : days   : STRARR output list of sorted, unique days in format yymmdd
;
; Opt. Outputs: None.
;
; Keywords    : SC     : STRING defining S/C :'1','2','3','4'
;               TSTART : STRING start date in form yymmdd.
;               TSTOP  : STRING stop date in form yymmdd.
;               HK     : FLAG indicating should only process HK files.
;               BM     : FLAG indicating should only process BM files.
;               NM     : FLAG indicating should only process NM files.
;
; Written     :	Version 0.0, Martin Carter, RAL, 6/1/01
;
; Modified    : Version 0.1, 27/1/01
;                 Added version options.
;               Version 0.2, 11/2/01
;                 Added batch mode.
;                 Changed sc selection.
;               Version 0.3, 2/3/01
;                 Added DAYS keyword.
;           	Version 0.4, 9/3/01
;                 Added TSTART and TSTOP keywords.
;                 Removed FILTER.
;               Version 0.5, 23/03/01
;                 Changed to rcs_sort_days.
;               Version 0.6, 25/05/01
;                 Added HK keyword.
;               Version 0.7, 31/05/01
;                 Added BM and NM keywords.
;               Version 0.8, 15/08/01
;                 Dealt with lower case CDROMs of the form /CDJB/010702_1_2A/cluster1/hkd_1/010702rh.1a1
;               Version 0.9, 22/08/01
;                 Allowed more than one root DATA directory.
;               Version 1.0, 17/01/03
;                 Changed routine to function returning days.
;               Version 1.1, 09/10/03
;                 Used ies get dir.
;               Version 1.2, 09/01/04
;                 Fixed minor bug if count=0 and added printout.
;
; Version     :	Version 1.2, 09/01/04
;-
;**********************************************************

FUNCTION rcs_add_cd, ids, SC=sc, TSTART=tstart, TSTOP=tstop, HK=hk, NM=nm, BM=bm

  ; initialize days

  days = ''

  ; get S/C required

  IF NOT KEYWORD_SET(sc) THEN sc = rcs_sc_selection(ids)

  ; test if root directory defined
  ; root directory is the path of the directory containing the CDs ie /cdrom
  ; this routine uses the CD volume labels to determine which days are present

  root = ies_get_dir('DATA',/RAPID)

  IF root EQ '' THEN BEGIN

    rcs_message, 'WARNING, NO DATA DIRECTORY DEFINED FOR SC'+sc, /INFORMATIONAL

    RETURN, days

  ENDIF

  ; get list of root directories

  roots = break_path(root,/NOCURRENT)

  ; get list of all sub-directories

  subdirs = ''

  FOR r = 0, N_ELEMENTS(roots)-1 DO subdirs = [subdirs,rcs_find_subdirs(roots[r])]

  subdirs = subdirs[1:*]

  ; directory format is \root\yymmdd_1_1A/CLUSTERn/NSD_n/yymmddRN.1An

  ; find all days

  list = INTARR(N_ELEMENTS(subdirs))

  FOR k = 0, N_ELEMENTS(subdirs)-1 DO IF STRLEN(strip_dirname(subdirs[k])) EQ 11 THEN list[k] = 1

  list = WHERE( list NE 0, count)

  IF count EQ 0 THEN BEGIN

    rcs_message, 'WARNING, NO VALID DAYS IN DATA DIRECTORY FOR SC'+sc, /INFORMATIONAL

    RETURN, days

  ENDIF

  subdirs = subdirs[list]

  ; select and sort days
  ; keep version labels for ordering
  ; CDROM labels have form yymmdd_1_2A, yymmdd_2_2A ie, yymmdd_i_nv
  ; want to sort these by yymmdd, v, i
  ; create name yymmddvi_nv

  subdir_sort = strip_dirname(subdirs)

  FOR k = 0, N_ELEMENTS(subdirs)-1 DO BEGIN

    temp = subdir_sort[k]

    STRPUT, temp, STRMID(subdir_sort[k],10,1), 6

    subdir_sort[k] = temp

  ENDFOR

  list = rcs_sort_days ( subdir_sort, TSTART=tstart, TSTOP=tstop, COUNT=count)

  ; check count

  IF count EQ 0 THEN rcs_no_days, sc, TSTART=tstart, TSTOP=tstop ELSE subdirs = subdirs[list]

  ; define types and codes

  IF KEYWORD_SET(hk) THEN BEGIN

    types = ['HKD']

    codes = ['RH']

    indices = [2]

  ENDIF ELSE IF KEYWORD_SET(nm) THEN BEGIN

    types = ['NSD']

    codes = ['RN']

    indices = [0]

  ENDIF ELSE IF KEYWORD_SET(bm) THEN BEGIN

    types = ['BSD']

    codes = ['RB']

    indices = [1]

  ENDIF ELSE BEGIN

    types = ['NSD','BSD','HKD']

    codes = ['RN','RB','RH']

    indices = [0,1,2]

  ENDELSE

  ; loop through file streams adding file names

  FOR n = 0, N_ELEMENTS(types)-1 DO BEGIN

    ; get files

    files = ''

    FOR k = 0, count-1 DO BEGIN

      ; get subdirectory name

      subdir = strip_dirname(subdirs[k])

      ; get day

      day = STRMID(subdir,0,6)

      ; get CDROM no

      num = STRMID(subdir,7,1)

      ; get version

      ver = STRMID(subdir,10,1)

      ; get directory name

      adddir = concat_dir('CLUSTER'+sc,types[n]+'_'+sc)

      adddir = concat_dir(adddir,day+codes[n]+'.'+num+ver+sc)

      ; get file

      file = concat_dir(subdirs[k],adddir)

      ; check file exists

      dummy = FINDFILE(file,COUNT=fcount)

      IF fcount NE 1 THEN BEGIN

        file = concat_dir(subdirs[k],STRLOWCASE(adddir))

        dummy = FINDFILE(file,COUNT=fcount)

      ENDIF

      IF fcount EQ 1 THEN BEGIN

        files = [files,file]

        days = [days,day]

      ENDIF ELSE BEGIN

        rcs_message, 'WARNING, FILE NOT FOUND : '+file, /INFORMATIONAL

      ENDELSE

    ENDFOR

    ; add files

    IF N_ELEMENTS(files) GT 1 THEN BEGIN

      files = files[1:*]

      ; set file list

      rcs_set_files, indices[n], files

    ENDIF

  ENDFOR

  IF N_ELEMENTS(days) GT 1 THEN BEGIN

    days = days[1:*]

    list = rcs_sort_days ( days)

    days = days[list]

  ENDIF

  RETURN, days

END
