;**********************************************************
;+
; Project     : CCLUSTER II - RAPID
;
; Name        : RCS_DDS_HEADER
;
; Purpose     : Extract date/time, and type info from ESOC header
;
; Explanation : This function is used to decode the 15 byte header
;             : included at the start of each telemetry record.
;             : The information includes the date/time of the
;             : packet as calculated by the ground segment from
;             : the S/C on-board clock, the packet size and
;             : information on the data source (S/C, inst. etc).
;
; Use         : < res = decode_header(header) >
;
; Inputs      : header : BYTARR[15] array containing the header
;
; Opt. Inputs : None.
;
; Outputs     : A structure containing the decoded header
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Calls       : rcs_cds_to_tai.
;
; Common      : None.
;
; Restrictions: The supplied header array must contain a valid
;             : 15 byte ESOC header.
;
; Side effects: None.
;
; Category    : Data Access
;
; Prev. Hist. : Based on original work for Cluster I.
;
; Written     : Version 0.0, Chris Perry, RAL, 10/08/1999
;
; Modified    : Version 0.1. MKC, 11/1/00
;               Version 0.2, 10/2/01
;                 Changed PACKET_HEADER structure.
;               Version 0.3, 14/03/01
;                 Removed ids from rcs_text and this argument list.
;
; Version     :	Version 0.3, 14/03/01
;-
;**********************************************************

FUNCTION rcs_dds_header, header, info

	; *** Extract information from the ESOC GS header ***
	; *** These headers are attached to all packets   ***

    packet_time = (rcs_cds_to_tai(header[0:7]))[0]

    packet_source = header[8]

    packet_length = rcs_long([REVERSE(header[9:11]),0b])

    packet_scid = header[12] AND '0F'X

    packet_gsid = ISHFT(header[12] AND 'F0'X,-4)

    packet_dtst = header[13]

    packet_timq = header[14] AND '0F'X

    packet_tasi = ISHFT(header[14] AND 'F0'X,-4)

    ; display packet info

    IF 0 THEN BEGIN

      rcs_text, [$
        'packet header time   = '+anytim2cal(packet_time),$
        'packet header source = '+packet_source,$
        'packet header length = '+packet_length,$
        'packet header scid   = '+packet_scid,$
        'packet header dtst   = '+packet_dtst,$
        'packet header timq   = '+packet_timq,$
        'packet header tasi   = ']

    ENDIF

    ; return header structure

    RETURN, {PACKET_HEADER, info:info, time:packet_time, length:packet_length}

END
