;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_FIND_DDS
;
; Purpose     :	Finds non-zero RAPID DDS data.
;
; Explanation : Reads header from specified file stream.
;               Finds non-zero data and stores header.
;
; Use         : < result = RCS_FIND_DDS(ids, info, n) >
;
; Inputs      : ids  : STRUCTURE containing widget IDs
;               info : STRUCTURE array containing file info
;               n    : INT index into info for current file
;
; Opt. Inputs : None.
;
; Outputs     :    0 : if more data needed
;                  2 : next file needed
;                  3 : all file ends reached or earliest header file end reached
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 17/9/00
;
; Modified    :	Version 0.1, 11/2/01
;                 Moved packet_header to COMMON.
;                 Removed SILENT keyword.
;                 Changed to silent routine.
;
; Version     :	Version 0.1, 12/2/01
;
; Version     :	Version 0.1, 11/2/01
;-
;**********************************************************

FUNCTION rcs_find_dds, ids, info, n

  ; set up io error response

  ON_IOERROR, io_error

  ; have just read data and zeroed header and found data contains zeroes
  ; want to note current position so can backtrack read n headers

  result = FSTAT(info[n].unit)

  cur_ptr = result.cur_ptr

  ; get packet header

  packet_header = rcs_get_dds_header ( ids, info, n)

  ; remove current header from store

  rcs_set_header, n, 0

  ; deal with no data

  IF NOT KEYWORD_SET(packet_header) THEN RETURN, 0

  ; initialize jump size

  njump = 256L

  ; set up array to hold packet

  packet = BYTARR(packet_header.length)

  WHILE 1 DO BEGIN

    ; try to jump forward njump packets and retrieve data

    remainder = result.size - cur_ptr

    ; check if sufficient data

    IF remainder GE njump*(15+packet_header.length) THEN BEGIN

      POINT_LUN, info[n].unit, cur_ptr + njump*(15+packet_header.length)-packet_header.length

      ; read packet

      READU, info[n].unit, packet

      ; check out packet contents

      list = WHERE(packet NE 'FF'XB AND packet NE '00'XB AND packet NE 'C0'XB, count)

      IF count NE 0 THEN BEGIN

        ; reduce jump size

        njump = njump/2

        IF njump EQ 0 THEN BEGIN

          ; backtrack to before header

          POINT_LUN, info[n].unit, cur_ptr

          RETURN, 0

        ENDIF

      ENDIF ELSE BEGIN

        ; update pointer

        cur_ptr = cur_ptr + njump*(15+packet_header.length)

      ENDELSE

    ENDIF ELSE BEGIN

      ; decrease jump size

      njump = njump/2

      ; check if reached limit

      IF njump EQ 0 THEN BEGIN

        ; backtrack to before header

        POINT_LUN, info[n].unit, cur_ptr

        RETURN, 0

      ENDIF

    ENDELSE

  ENDWHILE

  ; io error on trying to read next data from file

  io_error : rcs_message, !ERR_STRING

END
