;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_DAT_DATA
;
; Purpose     :	Reads RAPID DAT data packet.
;
; Explanation : Reads RAPID DAT data packet.
;
; Use         : < result = RCS_GET_DAT_DATA(ids, info, n, packet_header) >
;
; Inputs      : ids  : STRUCTURE containing widget IDs
;               info : STRUCTURE array containing file info
;               n    : INT index into info for current file
;               packet_header : STRUCTURE containing packet info
;
; Opt. Inputs : None.
;
; Outputs     :    0 : if more data needed
;                  1 : if acquired sufficient data
;                  2 : insufficient data in file to complete packet
;                 -1 : if zero data found
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 20/1/01
;
; Modified    : Version 0.1, 24/1/01
;                 Added check for unchanged HK frame counter.
;               Version 0.2, 11/2/01
;                 Moved packet_header to COMMON.
;                 Removed SILENT keyword.
;                 Changed to silent routine.
;               Version 0.3, 13/03/01
;                 Changed use of silent flag.
;               Version 0.4, 14/03/01
;                 Removed ids from rcs_add_to_common argument list.
;               Version 0.5, 12/04/01
;                 Corrected bug preventing compilation.
;               Version 0.6, 12/02/02
;                 Changed to make more robust against BYTEORDER problems in DAT file.
;
; Version     :	Version 0.6, 12/02/02
;-
;**********************************************************

FUNCTION rcs_get_dat_data, ids, info, n, packet_header

  ; set up io error response

  ON_IOERROR, io_error

  ; check if end of file

  IF EOF(info[n].unit) THEN RETURN, 2

  ; set up read variables

  length0 = 0L

  length1 = 0L

  packet = BYTARR(packet_header.length)

  READU, info[n].unit, length0, packet, length1

  IF length0 NE packet_header.length THEN BEGIN

    BYTEORDER, length0, /LSWAP

    BYTEORDER, length1, /LSWAP

  ENDIF

  IF length0 NE length1 OR length0 NE packet_header.length THEN BEGIN

    rcs_message, 'WARNING, INVALID FILE FORMAT'

  ENDIF

  ; remove current header

  rcs_set_header, n, 0

  ; increment bytes read

  info[n].bytes= info[n].bytes + packet_header.length

  ; set time of last packet read from this stream and bytes

  IF NOT rcs_get_silent() THEN BEGIN

    IF WIDGET_INFO(ids.in[n].time,/VALID_ID) THEN BEGIN

      WIDGET_CONTROL, ids.in[n].time, SET_VALUE=anytim2cal(packet_header.time,/MSEC)

      WIDGET_CONTROL, ids.in[n].bytes, SET_VALUE=rcs_strtrim(info[n].bytes)

    ENDIF

  ENDIF

  ; check out packet contents

  list = WHERE(packet NE 'FF'XB AND packet NE '00'XB AND packet NE 'C0'XB, count)

  IF count EQ 0 THEN RETURN, -1

  ; if HK stream and frame counter unchanged then do not store data

  IF ids.in[n].type EQ 'HK' THEN BEGIN

    ; get current frame counter

    fc = packet[0] AND '1F'X

    ; if frame counter unchanged then return saying need more data

    IF info[n].previous_fc EQ fc THEN RETURN, 0

    ; update frame counter

    info[n].previous_fc = fc

  ENDIF

  ; add data block to COMMON block

  RETURN, rcs_add_to_common(packet_header, packet)

  ; io error on trying to read next data from file

  io_error : rcs_message, !ERR_STRING

END
