;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_DAT_HEADER
;
; Purpose     :	Obtains DAT packet header from file stream.
;
; Explanation : Checks unit number to see if file open.
;               Checks if have stored packet_header.
;               If not, tries to read packet header.
;               If successful stores packet header.
;               If not returns 0.
;               Returns packet header,
;
; Use         : < packet_header = rcs_get_dat_header(ids, info. n) >
;
; Inputs      : ids  : STRUCTURE containing widget IDs
;               info : STRUCTURE array containing file info
;               n    : INT index into info for current file
;
; Opt. Inputs : None.
;
; Outputs     : packet_header : STRUCTURE containing packet info
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 19/1/01
;
; Modified    : Version 0.1, 10/2/01
;                 Changed PACKET_HEADER structure.
;               Version 0.2, 11/2/01
;                 Moved packet_header to COMMON.
;                 Removed SILENT keyword.
;                 Changed to silent routine.
;               Version 0.3, 13/03/01
;                 Changed use of silent flag.
;               Version 0.4, 14/03/01
;                 Changed so cannot use DIALOG_rcs_message in batch mode.
;                 Removed ids from rcs_text argument list.
;               Version 0.5, 12/02/02
;                 Changed to make more robust against BYTEORDER problems in DAT file.
;
; Version     :	Version 0.5, 12/02/02
;-
;**********************************************************

FUNCTION rcs_get_dat_header, ids, info, n

  ; set up io error response

  ON_IOERROR, io_error

  ; check if this data stream open

  IF info[n].unit EQ 0 THEN RETURN, 0

  ; get current header
  ; try widget first

  packet_header = rcs_get_header(n)

  ; if header not read already then read it

  IF NOT KEYWORD_SET(packet_header) THEN BEGIN

    ; check if end of file

    IF EOF(info[n].unit) THEN RETURN, 0

    ; set up read variables

    length0 = 0L

    length1 = 0L

    edb_utc = '123456789012345678901234'

    ndat = 0L

    edb_no = 0L

    edb_mode = 0L

    ; 1st record

    READU, info[n].unit, length0, edb_utc, ndat, edb_no, edb_mode, length1

    time = anytim2tai(edb_utc)

    IF length0 NE 36 THEN BEGIN

      BYTEORDER, length0, /LSWAP

      BYTEORDER, ndat, /LSWAP

      BYTEORDER, edb_no, /LSWAP

      BYTEORDER, edb_mode, /LSWAP

      BYTEORDER, length1, /LSWAP

    ENDIF

    IF length0 NE length1 OR length0 NE 36 THEN BEGIN

      rcs_message, 'WARNING, INVALID FILE FORMAT'

    ENDIF

    ; establish data type from file name

    packet_info = rcs_get_packet_info(info[n].file)

    ; set up type if not HK

    IF packet_info.type EQ '' THEN BEGIN
      CASE edb_mode OF
        0: packet_info.type = 'CCSNM'
        1: packet_info.type = 'CCSBM1'
        2: packet_info.type = 'CCSBM2'
        3: packet_info.type = 'CCSBM3'
        ELSE : rcs_message, 'INVALID MODE :'+STRING(edb_mode)
      ENDCASE
    ENDIF

    ; set up packet header structure

    packet_header = {PACKET_HEADER, info:packet_info, time:time, length:ndat}

    ; save current header

    rcs_set_header, n, packet_header

    ; check time increases

    IF packet_header.time LE info[n].previous_time THEN BEGIN

      message = ['DATA ERROR : INVALID PACKET TIME ON STREAM '+ids.in[n].type, $
                   'CURRENT :'+anytim2cal(packet_header.time), $
                   'PREVIOUS :'+anytim2cal(info[n].previous_time)]

      rcs_text, message

      IF NOT KEYWORD_SET(rcs_get_auto()) THEN $
        IF STRUPCASE(DIALOG_MESSAGE( /QUESTION, [message, 'CONTINUE ?'] )) EQ 'NO' THEN $
          rcs_message, 'INVALID TIME ERROR',/TRACEBACK

    ENDIF

    info[n].previous_time = packet_header.time

    ; set time of packet waiting in this stream

    IF NOT rcs_get_silent() THEN BEGIN

      IF WIDGET_INFO(ids.in[n].time,/VALID_ID) THEN $
        WIDGET_CONTROL, ids.in[n].time, SET_VALUE=anytim2cal(packet_header.time,/MSEC), /APPEND

    ENDIF

    ; check packet_header

    IF packet_header.length LE 0 THEN rcs_message, 'INVALID PACKET HEADER', /IMPLEMENTATION

  ENDIF

  RETURN, packet_header

  ; io error on trying to read next data from file

  io_error : rcs_message, !ERR_STRING

END
