;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_DATA
;
; Purpose     :	Reads RAPID GSEOS or DDS data.
;
; Explanation : Reads a number of data blocks from a set of files.
;               If end of files reached then prompts for new file(s).
;               Extracts data to COMMON blocks.
;               Should be called with reset first time used.
;               NB GSEOS stamp is associated with ID.
;               Potentially could have more than one ID associated with the
;               same header.
;
; Use         : < result = RCS_GET_DATA(ids) >
;
; Inputs      : ids  : STRUCTURE containing widget IDs
;
; Opt. Inputs : None.
;
; Outputs     : 0 : EOF FOUND.
;               1 : GOT DATA OK.
;
; Opt. Outputs: None.
;
; Keywords    : RESET : FLAG indicating reset.
;               SKIP  : either FLAG or UTC time.
;               WEB   : FLAG indicating WEB directory used for output.
;
; Written     :	Version 0.0, Martin Carter, RAL, 17/9/00
;
; Modified    :	Version 0.1, 13/12/00
;                 info now contains previous time explicitly
;                 Modified check on type of data to use with rcs_find_dds.
;                 Used ids.type tag.
;                 Added rcs_get_dat.
;                 Store info between calls and obsolete corresponding uvalues.
;               Version 0.2, 10/2/01
;                 Changed silent and batch to common blocks.
;               Version 0.3, 21/2/01
;                 Renamed batch --> auto
;                 Moved graph keyword.
;               Version 0.4, 14/03/01
;                 Removed ids from rcs_batch_process argument list.
;               Version 0.5, 18/07/01
;                 Modified to use extended IF names.
;               Version 0.6, 13/03/02
;                 Changed routine rcs_if_day name.
;               Version 0.7, 29/07/02
;                 Removed ids from some argument lists.
;               Version 0.8, 03/10/03
;                 Added WEB keyword.
;
; Version     :	Version 0.8, 03/10/03
;-
;**********************************************************

FUNCTION rcs_get_data, ids, RESET=reset, SKIP=skip, WEB=web

  ; first time initializations
  ; NB cannot rely on COMMON block being uninitialized

  IF KEYWORD_SET(reset) THEN BEGIN

    ; reset block info
    ; use block widget uvalue
    ; only used by GSEOS

    IF ids.type EQ 'GSEOS' THEN WIDGET_CONTROL, ids.block, SET_UVALUE=0

    RETURN, 0

  ENDIF

  ; restore info on current state of each file type

  info = rcs_get_info()

  ; open top file of file list if not already open

  IF NOT KEYWORD_SET(rcs_get_auto()) THEN BEGIN

    FOR k = 0, N_ELEMENTS(info)-1 DO BEGIN

      ; check if file open yet
      ; file may be part read and wish to continue or not yet opened
      ; dont want to call rcs_next_file if opened

      IF info[k].unit EQ 0 THEN BEGIN

        ; reset block info

        IF ids.type EQ 'GSEOS' THEN WIDGET_CONTROL, ids.block, SET_UVALUE=0

        ; open next file

        rcs_next_file, ids, info, k

      ENDIF

    ENDFOR

  ENDIF ELSE BEGIN

    ; if any files open then continue
    ; if no files open then try to open
    ; if IF already exist then process and return

    ; open next files and return

     res = rcs_next_files(ids, info, WEB=web)

     IF res LT 2 THEN GOTO, return

  ENDELSE

  ; check if any files defined

  list = WHERE(info.nfiles GT 0, count)

  IF count EQ 0 THEN BEGIN

    res = 0

    GOTO, return

  ENDIF

  IF ids.type EQ 'DDS' AND KEYWORD_SET(skip) THEN BEGIN

    ; this needs further modification to allow non-timed skip

    FOR k = 0, N_ELEMENTS(ids.in)-1 DO BEGIN

      IF N_TAGS(skip) EQ 2 THEN BEGIN

        IF info[k].unit NE 0 THEN BEGIN

          rcs_skip_dds, ids, info, k, skip

        ENDIF

      ENDIF

    ENDFOR

    ; turn off skip button

    WIDGET_CONTROL, ids.skiptime, SET_UVALUE=0

    WIDGET_CONTROL, ids.skipid, SET_BUTTON=0

    WIDGET_CONTROL, ids.skiptime, SET_VALUE=''

  ENDIF

  ; read block bodies

  WHILE 1 DO BEGIN

    ; possible results :
    ;   0 : processed data OK but not sufficient data yet
    ;   1 : processed data OK and sufficient data now
    ;   2 : reached file end want next file
    ;   3 : reached file end, wait for more data

    CASE ids.type OF

      'GSEOS' : result = rcs_get_gseos( ids, info, n )

      'DDS' : result = rcs_get_dds ( ids, info, n)

      'DAT' : result = rcs_get_dat ( ids, info, n)

      ELSE : rcs_message, /IMPLEMENTATION

    ENDCASE

    ; check result

    CASE result OF

     -1 : IF ids.type EQ 'DDS' AND (ids.in[n].type EQ 'NM' OR ids.in[n].type EQ 'HK') THEN BEGIN

            ; NB BM has variable sized data packets

            result = rcs_find_dds( ids, info, n )

          ENDIF

      0 : ; insufficient data acquired : continue

      1 : BEGIN

            res =  1 ; sufficient data acquired : return unblocked code

            GOTO, return

          END

      3 : BEGIN

            res = 0 ; all APPEND files blocked

            GOTO, return

          END

      2 : BEGIN

            ;  need new file

            ; io error or not availiable on trying to read next data

            ; check if end of file
            ; NB this checks that DDS files do not have partial records

            IF EOF(info[n].unit) THEN BEGIN

              ; reset block info

              IF ids.type EQ 'GSEOS' THEN WIDGET_CONTROL, ids.block, SET_UVALUE=0

              ; check if auto mode

              IF KEYWORD_SET(rcs_get_auto()) THEN BEGIN

                ; get current file

                current_file = info[n].file

                ; close currentfile
                ;  nb updates info and files displayed

                rcs_close_file, info, n

                ; check if any other files have the same date or version

                flags = rcs_check_files ( ids, info, current_file)

                ; check if file on same stream and open

                IF flags[n] THEN rcs_next_file, ids, info, n

                ; if no other files for this day then process data else continue

                IF TOTAL(flags) EQ 0 THEN BEGIN

                  ; process all stored data

                  rcs_get_sc_edbs, ids

                  rcs_get_hk_edbs, ids

                  ; save IF data

                  rcs_save_if, DAY=rcs_raw_day(current_file)

                  ; check if any data

                  sizes = rcs_get_sizes()

                  ; process data

                  IF sizes.ifsize NE 0 THEN $
                    rcs_batch_process, rcs_in_sc(current_file), rcs_in_day(ids,current_file), WEB=web

                  ; reset but leave files alone

                  rcs_reset, ids

                  ; return control

                  res = 1

                  GOTO, return

                ENDIF

              ENDIF ELSE BEGIN

                ; close file

                rcs_close_file, info, n

                ; get next file

                rcs_next_file, ids, info, n

                ; return if no file given

                IF info[n].nfiles EQ 0 THEN BEGIN

                  res = 0

                  GOTO, return

                ENDIF

              ENDELSE

            ENDIF ELSE rcs_message, 'FILE DATA ERROR'

          END

    ENDCASE

  ENDWHILE

  return :

  ; save info on current state of each file type

  rcs_set_info, info

  RETURN, res

END
