;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_DDS_DATA
;
; Purpose     :	Reads RAPID DDS data packet..
;
; Explanation : Checks if packet complete :
;               If complete :
;                 reads packet;
;                 checks contains real data else returns 0;
;                 adds packet to common block;
;                 returns 0 if more data needed
;                         1 if have sufficient data
;               If incomplete :
;                 closes and re-opens file;
;                 positions pointer to current position;
;                 returns 2.
;
; Use         : < result = RCS_GET_DDS(ids, info, n, packet_header) >
;
; Inputs      : ids  : STRUCTURE containing widget IDs
;               info : STRUCTURE array containing file info
;               n    : INT index into info for current file
;               packet_header : STRUCTURE containing packet info
;
; Opt. Inputs : None.
;
; Outputs     :    0 : if more data needed
;                  1 : if acquired sufficient data
;                  2 : insufficient data in file to complete packet
;                 -1 : if zero data found
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Calls       : rcs_add_to_common,
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    : RAPID COMMISSIONING SOFTWARE.
;
; Prev. Hist. :	Adapted from read_filter.
;
; Written     :	Version 0.0, Martin Carter, RAL, 8/1/00
;
; Modified    :	Version 0.1, 13/12/00
;                 Moved time check to rcs_get_dds_header
;                 Stored info between calls.
;               Version 0.2, 11/2/01
;                 Moved packet_header to COMMON.
;                 Removed SILENT keyword.
;                 Changed to silent routine.
;               Version 0.3, 13/03/01
;                 Changed use of silent flag.
;               Version 0.4, 14/03/01
;                 Removed ids from rcs_add_to_common argument list.
;
; Version     :	Version 0.4, 14/03/01
;-
;**********************************************************

FUNCTION rcs_get_dds_data, ids, info, n, packet_header

  ; set up io error response

  ON_IOERROR, io_error

  ; file may be being appended to
  ; dont attempt to read if not sufficient data

  result = FSTAT(info[n].unit)

  remainder = result.size - result.cur_ptr

  ; check if sufficient data

  IF remainder GE packet_header.length THEN BEGIN

    ; set up array to hold packet

    packet = BYTARR(packet_header.length)

    ; read packet

    READU, info[n].unit, packet

    ; remove current header

    rcs_set_header, n, 0

    ; increment bytes read

    info[n].bytes= info[n].bytes + packet_header.length

    ; set display

    IF NOT rcs_get_silent() THEN BEGIN

      IF WIDGET_INFO(ids.in[n].time,/VALID_ID) THEN BEGIN

        WIDGET_CONTROL, ids.in[n].time, SET_VALUE=anytim2cal(packet_header.time,/MSEC)

        WIDGET_CONTROL, ids.in[n].bytes, SET_VALUE=rcs_strtrim(info[n].bytes)

      ENDIF

    ENDIF

    ; check out packet contents

    list = WHERE(packet NE 'FF'XB AND packet NE '00'XB AND packet NE 'C0'XB, count)

    IF count EQ 0 THEN RETURN, -1

    ; add data block to COMMON block

    RETURN, rcs_add_to_common(packet_header, packet)

  ENDIF

  ; data packet not complete
  ; close file

  FREE_LUN, info[n].unit

  ; re-open file

  OPENR, unit, info[n].file, /GET_LUN

  ; save unit

  info[n].unit = unit

  ; position to start of data

  POINT_LUN, info[n].unit, result.cur_ptr

  ; return saying have reached end of file

  RETURN, 2

  ; io error on trying to read next data from file

  io_error : MESSAGE, !ERR_STRING

END
