;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_DDS_HEADER
;
; Purpose     :	Obtains DDS packet header from file stream.
;
; Explanation : Checks unit number to see if file open.
;               Checks if have stored packet_header.
;               If not, tries to read packet header.
;               If successful stores packet header.
;               If not returns 0.
;               Returns packet header,
;
; Use         : < packet_header = rcs_get_dds_header(ids, info. n) >
;
; Inputs      : ids  : STRUCTURE containing widget IDs
;               info : STRUCTURE array containing file info
;               n    : INT index into info for current file
;
; Opt. Inputs : None.
;
; Outputs     : packet_header : STRUCTURE containing packet info
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 8/1/00
;
; Modified    :	Version 0.1, 13/12/00
;                 Moved time check and time display into here.
;                 Changed printout for invalid packet time error to
;                 DIALOG_rcs_message.
;                 Signal error if non-increasing time.
;                 info now contains previous time explicitly
;                 Added keyword SILENT.
;                 Stored info between calls.
;               Version 0.2, 11/2/01
;                 Moved packet_header to COMMON.
;                 Removed SILENT keyword.
;                 Changed to silent routine.
;               Version 0.3, 13/03/01
;                 Changed use of silent flag.
;               Version 0.4, 14/03/01
;                 Changed so cannot use DIALOG_rcs_message in batch mode.
;                 Removed ids from rcs_text argument list.
;               Version 0.5, 09/05/01
;                 Tidied up processing of bad times. Error message was put out twice.
;
; Version     :	Version 0.5, 09/05/01
;-
;**********************************************************

FUNCTION rcs_get_dds_header, ids, info, n

  ; set up io error response

  ON_IOERROR, io_error

  ; check if this data stream open

  IF info[n].unit EQ 0 THEN RETURN, 0

  ; get current header
  ; try widget first

  packet_header = rcs_get_header(n)

  ; if header not read already then read it

  IF NOT KEYWORD_SET(packet_header) THEN BEGIN

    ; set up header array

    packet_header = BYTARR(15)

    ; file may be being appended to
    ; dont attempt to read if not sufficient data

    result = FSTAT(info[n].unit)

    remainder = result.size - result.cur_ptr

    IF remainder GE 15 THEN BEGIN

      READU, info[n].unit, packet_header

      ; establish data type from file name

      packet_info = rcs_get_packet_info(info[n].file)

      ; process packet header

      packet_header = rcs_dds_header(packet_header,packet_info)

      ; save current header

      rcs_set_header, n, packet_header

      ; check time increases
      ; NB get equal times quite frequently -- just issue warning
      ; NB Cannot guarantee that one ofthe streams NM, BM, HK will not be
      ; held up. Thus HK stream and SC stream may not increase in time order.
      ; Also BM and NM are multiplexed into SC stream. There is no guarantee
      ; that the NM and BM packets will be in time order.

      IF packet_header.time LE info[n].previous_time THEN BEGIN

        message = ['DATA ERROR : INVALID PACKET TIME ON STREAM '+ids.in[n].type, $
                   'CURRENT :'+anytim2cal(packet_header.time), $
                   'PREVIOUS :'+anytim2cal(info[n].previous_time)]

        rcs_text, message

        IF NOT KEYWORD_SET(rcs_get_auto()) THEN BEGIN

          IF packet_header.time LT info[n].previous_time THEN $
            IF STRUPCASE(DIALOG_MESSAGE( /QUESTION, [message, 'CONTINUE ?'] )) EQ 'NO' THEN $
            rcs_message, 'INVALID TIME ERROR',/TRACEBACK

        ENDIF

      ENDIF

      ; set time of packet waiting in this stream

      info[n].previous_time = packet_header.time

      IF NOT rcs_get_silent() THEN BEGIN

        IF WIDGET_INFO(ids.in[n].time,/VALID_ID) THEN $
          WIDGET_CONTROL, ids.in[n].time, SET_VALUE=anytim2cal(packet_header.time,/MSEC), /APPEND

      ENDIF

      ; check packet_header

      IF packet_header.length LE 0 THEN rcs_message, 'INVALID PACKET HEADER', /IMPLEMENTATION

    ENDIF ELSE BEGIN

      ; header packet not complete
      ; close file

      FREE_LUN, info[n].unit

      ; re-open file

      OPENR, unit, info[n].file, /GET_LUN

      ; save unit

      info[n].unit = unit

      ; position to start of data

      POINT_LUN, info[n].unit, result.cur_ptr

      ; return saying have reached end of file

      RETURN, 0

    ENDELSE

  ENDIF ELSE BEGIN

    ; check packet_header

    IF packet_header.length LE 0 THEN rcs_message, 'INVALID PACKET HEADER', /IMPLEMENTATION

  ENDELSE

  RETURN, packet_header

  ; io error on trying to read next data from file

  io_error : rcs_message, !ERR_STRING

END
