;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_GET_GSEOS
;
; Purpose     :	Unpacks RAPID GSEOS data.
;
; Explanation : Reads and processes a data packet from GSEOS file.
;               Data packet may be a block header or data block.
;               Extracts data to COMMON blocks.
;               Returns flag indicating result.
;
; Use         : < result = RCS_GET_GSEOS(ids, unit, block_info, time, packet_length) >
;
; Inputs      :
;
; Opt. Inputs : None.
;
; Outputs     :    0 : if block header or more data needed
;                  1 : if acquired sufficient data
;                  2 : if io error (most likely end of file)
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 8/1/00
;
; Modified    :	Version 0.1, 13/12/00
;                 Added DIALOG_rcs_message for backwards times.
;                 Signal error if non-increasing time.
;                 info now contains previous time explicitly
;                 Use info.previous_time rather than MAX(info.time)
;                 Stored info between calls.
;               Version 0.2, 12/2/01
;                 Removed SILENT keyword.
;                 Changed to silent routine.
;               Version 0.3, 13/03/01
;                 Changed use of silent flag.
;               Version 0.4, 14/03/01
;                 Changed so cannot use DIALOG_rcs_message in batch mode.
;                 Removed ids from rcs_text and rcs_add_to_common argument lists.
;
; Version     :	Version 0.4, 14/03/01
;-
;**********************************************************

FUNCTION rcs_get_gseos, ids, info, n

  ; set value

  n = 0

  ; set up io error response

  ON_IOERROR, io_error

  ; get block info

  WIDGET_CONTROL, ids.block, GET_UVALUE=block_info

  ; initialize

  id = BYTARR(2)

  READU, info.unit, id

  CASE STRING(id) OF

    'DE' : BEGIN

           ; read block header

           block_header = BYTARR(30)

           READU, info.unit, block_header

           ; save block info

           IF N_TAGS(block_info) EQ 0 THEN $
             block_info = {BLOCK_INFO, stamp:-1, id:rcs_fix(block_header[0:1]), header:STRING(block_header[2:27])} $
           ELSE $
             block_info = [block_info, {BLOCK_INFO, stamp:-1, id:rcs_fix(block_header[0:1]), header:STRING(block_header[2:27])}]

           WIDGET_CONTROL, ids.block, SET_UVALUE=block_info

           rcs_text, 'BLOCK HEADER = '+STRING(block_header[2:27])

           ; say still going

           RETURN, 0

         END

    'TA' : BEGIN

           ; read block body info

           block_body_info = BYTARR(10)

           READU, info.unit, block_body_info

           ; check only one file block ID

           file_block_id = rcs_fix(block_body_info[0:1])

           IF N_TAGS(block_info) EQ 0 THEN rcs_message, ' INVALID FILE BLOCK BODY'

           list = WHERE ( block_info.id eq file_block_id, count)

           IF count NE 1 THEN rcs_message, ' INVALID FILE BLOCK ID'

           stamp_number = rcs_fix(block_body_info[2:3])

           ; check stamp no. for id 1 more than previous
           ; NB previous stamp no initially set to -1
           ; NB GSEOS stamp is associated with ID.
           ; Potentially could have more than one ID associated with the
           ; same header.

           IF block_info[list[0]].stamp NE -1 AND stamp_number NE block_info[list[0]].stamp+1 THEN $
             rcs_text, 'INVALID STAMP NUMBER : '+STRING(block_info[list[0]].stamp)+','+STRING(stamp_number)

           block_info[list[0]].stamp = stamp_number

           WIDGET_CONTROL, ids.block, SET_UVALUE=block_info

           ; get data size

           wsize = rcs_fix(block_body_info[4:5])

           ; read block data

           data = BYTARR(wsize)

           READU, info.unit, data

           ; get time (ensure scalar)

           packet_time = anytim2tai('1/JAN/1970') + rcs_long(block_body_info[6:9])

           ; remove extra header and set up header structure

           header = rcs_gseos_header(ids, block_info[list[0]].header, packet_time, data, info.file)

           ; read last two bytes

           wlen = BYTARR(2)

           READU, info.unit, wlen

           IF rcs_fix(wlen) NE wsize + 14 THEN rcs_message, 'INVALID WLEN, WSIZE = '+STRING(rcs_fix(wlen))+STRING(wsize)

           ; check time monotonically increases

           IF header.time LE info.previous_time THEN BEGIN

             message = ['DATA ERROR : INVALID PACKET TIME', $
                        'CURRENT :'+anytim2cal(header.time), $
                        'PREVIOUS :'+anytim2cal(info.previous_time)]

             rcs_text, message

		     ; find that for gseos data have many packets with same time stamp
		     ; so cannot use strictly monotonic check

             IF NOT KEYWORD_SET(rcs_get_auto()) THEN $
               IF header.time LT info.previous_time THEN $
                 IF STRUPCASE(DIALOG_MESSAGE( /QUESTION, [message, 'CONTINUE ?'] )) EQ 'NO' THEN $
                   rcs_message, 'INVALID TIME ERROR',/TRACEBACK

           ENDIF

           ; check out packet contents

           list = WHERE(data NE 'FF'XB AND data NE '00'XB AND data NE 'C0'XB, count)

           IF count EQ 0 THEN RETURN, 0

           ; update info

           info.previous_time = header.time

           info.bytes = info.bytes + header.length

           ; update display

           IF NOT rcs_get_silent() THEN BEGIN

             IF WIDGET_INFO(ids.in.time,/VALID_ID) THEN BEGIN

               WIDGET_CONTROL, ids.in.time, SET_VALUE=anytim2cal(header.time)

               WIDGET_CONTROL, ids.in.bytes, SET_VALUE=rcs_strtrim(info.bytes)

             ENDIF

           ENDIF

           ; add data block to COMMON block

           RETURN, rcs_add_to_common(header, data)

         END

    ELSE : rcs_message, 'INVALID ID'

  ENDCASE

  io_error : RETURN, 2

END
