;**********************************************************
;+
; Project     : CCLUSTER II - RAPID
;
; Name        : RCS_GSEOS_FILE_HEADER
;
; Purpose     : Extract date/time, and type info from ESOC header
;
; Explanation : This function is used to decode the 15 byte header
;             : included at the start of each telemetry record.
;             : The information includes the date/time of the
;             : packet as calculated by the ground segment from
;             : the S/C on-board clock, the packet size and
;             : information on the data source (S/C, inst. etc).
;
; Use         : < res = decode_header(header) >
;
; Inputs      : header : BYTARR[15] array containing the header
;
; Opt. Inputs : None.
;
; Outputs     : A structure containing the decoded header
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Calls       : rcs_cds_to_tai.
;
; Common      : None.
;
; Restrictions: The supplied header array must contain a valid
;             : 15 byte ESOC header.
;
; Side effects: None.
;
; Category    : Data Access
;
; Prev. Hist. : Based on original work for Cluster I.
;
; Written     : Version 0.0, Chris Perry, RAL, 10/08/1999
;
; Modified    : Version 0.1. MKC, 11/1/00
;               Version 0.2, 14/03/01
;                 Removed ids from rcs_text and this argument list.
;
; Version     :	Version 0.2, 14/03/01
;-
;**********************************************************

PRO rcs_gseos_file_header, unit

  ON_IOERROR, error

  ; read file header

  file_header = BYTARR(32)

  READU, unit, file_header

  ; extract file header info

  id_0 = file_header[0]
  id_1 = file_header[1]
  id_string = STRING(file_header[2:4])
  flight_recorder_version = file_header[5:6]
  project_name = STRING(file_header[7:25])
  tai_file_time = anytim2tai('1/JAN/1970') + rcs_long(file_header[26:29])

  ; check file header

  IF id_0 NE 'EB'X THEN rcs_message, 'Bad ID 0'
  IF id_1 NE '90'X THEN rcs_message, 'Bad ID 1'
  IF id_string NE 'GSE' THEN rcs_message, 'Bad ID string'

  ; display file header data

  rcs_text, [$
    'Flight recorder version = '+STRING(flight_recorder_version,FORMAT='(Z4)'),$
    'Project Name = '+project_name,$
    'Time = '+anytim2cal(tai_file_time)]

  RETURN

  ; error

  error : rcs_message, !ERR_STRING

END

