;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_OPEN_FILE
;
; Purpose     :	Opens (and decompresses) file.
;
; Explanation : Opens file with check for gzipped file extension.
;               Checks for multiple (i.e compressed and uncompressed) versions of file.
;               Opens file with compressed option if necessary.
;               Returns logical unit number for file.
;               May optionally specify the file extension of the uncompressed file.
;               NB on PC file extension is overwritten by 'gz' when compressed.
;               If file extension is specified then if file not compressed checks that file has correct extension.
;
; Use         : < res = rcs_open_file(name,unit,file,EXTENSION=extension) >
;
; Inputs      : name : STRING filename with file extension (wild characters allowed)
;
; Opt. Inputs : None.
;
; Outputs     : res  : INT flag indicating file opened successfully or not.
;
; Opt. Outputs: unit : INT logical unit number.
;               file : STRING file name of opened file.
;
; Keywords    : EXTENSION : STRING[] file extensions of uncompressed file
;               QUIET     : FLAG suppresses printout.
;
; Written     :	Version 0.0, 11/07/02
;
; Modified    : Version 0.1, 31/07/03
;                 Added printout and QUIET keyword.
;                 Corrected bug : file_extension not defined.
;                 Corrected bug : if multiple files printout fails.
;               Version 0.2, 02/09/03
;                 If multiple files then use first.
;                 Corrected bug in printout.
;               Version 0.3, 18/09/03
;                 Allow multiple allowed extensions.
;
; Version     :	Version 0.3, 18/09/03
;-
;**********************************************************

FUNCTION rcs_open_file, name, unit, file, EXTENSION=extension, QUIET=quiet

  ON_IOERROR, ioerror

  ; get files with fileroot and any file extension

  files = FINDFILE(name)

  ; check if any files found

  IF files[0] EQ '' THEN RETURN, 0

  ; check if multiple files found

  IF N_ELEMENTS(files) GT 1 THEN BEGIN

    FOR k=0, N_ELEMENTS(files)-1 DO rcs_message, 'Multiple files found '+files[k], /CONTINUE

  ENDIF

  ; set up file name

  file = files[0]

  ; printout

  IF NOT KEYWORD_SET(quiet) THEN rcs_text, 'Opening file : '+file, /OVERRIDE

  ; check for gzipped file extension and open file

  file_extension = ies_get_file_extension(file)

  IF file_extension EQ 'gz' THEN BEGIN

    OPENR, unit, file, /GET_LUN, /COMPRESS

  ENDIF ELSE IF NOT KEYWORD_SET(extension) THEN BEGIN

    OPENR, unit, file, /GET_LUN

  ENDIF ELSE BEGIN

    FOR k = 0, N_ELEMENTS(extension)-1 DO IF file_extension EQ extension[k] THEN found = 1

    IF KEYWORD_SET(found) THEN BEGIN

      OPENR, unit, file, /GET_LUN

    ENDIF ELSE BEGIN

      rcs_message, 'Invalid file name : '+file, /CONTINUE

      RETURN, 0

    ENDELSE

  ENDELSE

  ; return OK

  RETURN, 1

ioerror : rcs_message, !ERR_STRING, /INFORMATIONAL

  ; NB unit=0 is not valid

  IF KEYWORD_SET(unit) THEN FREE_LUN, unit

  RETURN, 0

END
