;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_SKIP_DDS
;
; Purpose     :	Skips through RAPID DDS data.
;
; Explanation : Skips through DDS file until after specified time.
;               Checks uvalue to see if has a stored packet_header.
;               If stored header then if time after skip.time then
;               returns else deletes header and backtracks file to
;               before header.
;               Loops until header time after skip.time when it stores
;               packet header in uvalue and returns.
;               If reaches end of file before finding packet_header after
;               skip.time then returns and leaves uvalue unset.
;
; Use         : < rcs_skip_dds, ids, info, k, skip >
;
; Inputs      : ids  : STRUCTURE containing widget IDs
;               info : STRUCTURE array containing file info
;               k    : INT index into info
;               skip : FLAG or UTC structure
;
; Opt. Inputs : None,
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 17/9/00
;
; Modified    :	Version 0.1, 24/1/01
;                 Stored info between calls.
;               Version 0.2, 11/2/01
;                 Moved packet_header to COMMON.
;                 Removed SILENT keyword.
;                 Changed to silent routine.
;               Version 0.3, 13/03/01
;                 Changed use of silent flag.
;               Version 0.4, 14/03/01
;                 Removed ids from rcs_dds_header argument list.
;
; Version     :	Version 0.4, 14/03/01
;;-
;**********************************************************

PRO rcs_skip_dds, ids, info, k, skip

  ; set up io error response

  ON_IOERROR, io_error

  ; set up header array

  byte_header = BYTARR(15)

  ; get any stored header

  temp_header = rcs_get_header(k)

  IF KEYWORD_SET(temp_header) THEN BEGIN

    ; check packet_header

    IF temp_header.length LE 0 THEN rcs_message, 'INVALID PACKET HEADER', /IMPLEMENTATION

    ; check if skip time set

    IF N_TAGS(skip) EQ 2 THEN BEGIN

      ; check current time before skip time else return

      current = anytim2utc(temp_header.time)

      ; only compare time within day

      IF current.time GE skip.time THEN RETURN

    ENDIF

    ; backtrack

    result = FSTAT(info[k].unit)

    POINT_LUN, info[k].unit, result.cur_ptr - 15

    ; remove stored header

    rcs_set_header, ids, k, 0

  ENDIF

  ; if skip time set then read until past skiptime

  WHILE 1 DO BEGIN

    ; check for skip event

    res=WIDGET_EVENT(ids.skipid, BAD_ID=bad_id, /NOWAIT)

    IF res.id NE 0 THEN BEGIN

      ; sink button release

      res=WIDGET_EVENT(ids.skipid, BAD_ID=bad_id, /NOWAIT)

      ; turn off skip button

      WIDGET_CONTROL, ids.skiptime, SET_UVALUE=0

      ; turn off skip flag

      skip = 0

      RETURN

    ENDIF

    ; get header
    ; file may be being appended to
    ; dont attempt to read if not sufficient data

    result = FSTAT(info[k].unit)

    remainder = result.size - result.cur_ptr

    IF remainder GE 15 THEN BEGIN

      READU, info[k].unit, byte_header

      ; establish data type from file name

      packet_info = rcs_get_packet_info(info[k].file)

      ; check filename valid

      IF NOT KEYWORD_SET(packet_info) THEN rcs_message,'INVALID FILE NAME'

      ; process packet header

      packet_header = rcs_dds_header(byte_header,packet_info)

      ; check packet_header

      IF packet_header.length LE 0 THEN rcs_message, 'INVALID PACKET HEADER', /IMPLEMENTATION

      ; check if skip time set

      IF N_TAGS(skip) EQ 2 THEN BEGIN

        current = anytim2utc(packet_header.time)

        ; only compare time within day

        IF current.time GE skip.time THEN BEGIN

          ; this packet is after desired time -- store packet header

          rcs_set_header, k, packet_header

          RETURN

        ENDIF

      ENDIF

      remainder = remainder - 15

      IF remainder GE packet_header.length THEN BEGIN

        ; skip over packet data

        POINT_LUN, info[k].unit, result.cur_ptr + 15 + packet_header.length

        info[k].previous_time  = packet_header.time

        info[k].bytes = info[k].bytes + packet_header.length

        IF NOT rcs_get_silent() THEN BEGIN

          IF WIDGET_INFO(ids.in[k].time,/VALID_ID) THEN BEGIN

            WIDGET_CONTROL, ids.in[k].time, SET_VALUE=anytim2cal(packet_header.time,/MSEC)

            WIDGET_CONTROL, ids.in[k].bytes, SET_VALUE=rcs_strtrim(info[k].bytes)

          ENDIF

        ENDIF

      ENDIF ELSE BEGIN

        ; next packet not complete -- store packet header

        rcs_set_header, k, packet_header

        RETURN

      ENDELSE

    ENDIF ELSE BEGIN

      ; header packet not complete

      RETURN

    ENDELSE

  ENDWHILE

  ; io error on trying to read next data from file

  io_error : rcs_message, !ERR_STRING

END
