;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_BATCH
;
; Purpose     :	Manages RCS process in batch mode.
;
; Explanation : Allows full batch mode operation.
;
; Use         : < rcs_batch >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : DDS    : Flag indicating DDS data source.
;               DAT    : Flag indicating DAT data source.
;               SCI    : Flag indicating SCI data source.
;               GSEOS  : Flag indicating GSEOS data source.
;               TSTART : STRING start date in form yymmdd.
;               TSTOP  : STRING stop date in form yymmdd.
;               IFCFG  : STRING name of IF CFG file.
;               FORCE  : FLAG indicating should force processing of data even if
;                        IF data set already exists.
;               CLASSES: STRING array giving SCI classes to process
;               SILENT : FLAG indicating should not produce detailed output.
;               SC1    : Flag indicating should process SC1.
;               SC2    : Flag indicating should process SC2.
;               SC3    : Flag indicating should process SC3.
;               SC4    : Flag indicating should process SC4.
;               PROCESS_IF : Flag indicating should process directly from IF data sets.
;               REPROCESS_IF : Flag indicating should reprocess IF data sets to remove errors.
;               POLAR_4SC : Flag indicating should generate POLAR 4SC plot.
;               IES_4SC  : Flag indicating should generate IES 4SC plot.
;               IIMS_4SC : Flag indicating should generate IIMS 4SC plot.
;               PA_4SC   : Flag indicating should generate PA 4SC plot.
;               LINE_4SC : Flag indicating should generate LINE 4SC plot.
;               OCC_4SC  : Flag indicating should generate OCC 4SC plot.
;               HK     : FLAG indicating should only process HK files.
;               BM     : FLAG indicating should only process BM files.
;               NM     : FLAG indicating should only process NM files.
;               MCP_PROCESS : INT indicating should perform MCP processing.
;               TEMP_PROCESS : FLAG indicating should perform temperature processing.
;               SOURCE : STRING overriding default data source code ire w,z,DDS,SCI etc
;               REALTM : FLAG indicating should generate realtime plots
;               SURVEY : FLAG indicating should generate single SC survey plots
;               SW_VERSION : STRING overriding default sw version number
;
; Written     :	Version 0.0, 11/2/01
;
; Modified    :	Version 0.1, 26/2/01
;                 Changed batch tag to auto.
;           	Version 0.2, 9/3/01
;                 Added TSTART and TSTOP keywords.
;                 Added IFCFG keyword.
;                 Removed DAY_FILTER.
;                 Added RAPID results directory for 4SC results.
;                 Removed unused COMMON block.
;                 Changed use of silent flag.
;                 Added FORCE keyword.
;                 Added SC1, SC2, SC3, SC4.
;                 Added IIMS 4SC survey plot.
;               Version 0.3, 14/03/01
;                 Added SILENT keyword.
;               Version 0.4, 16/03/01
;                 Added LOADCT.
;               Version 0.5, 22/03/01
;                 Added PROCESS_IF.
;               Version 0.6, 08/04/01
;                 Added REPROCESS_IF keyword.
;                 Enforced batch_if = 4.
;               Version 0.7, 03/05/01
;                 Added POLAR 4SC plot.
;               Version 0.8, 09/05/01
;                 Added POLAR_4SC, IES_4SC, IIMS_4SC keywords.
;               Version 0.9, 25/05/01
;                 Added HK keyword.
;               Version 1.0, 31/05/01
;                 Added BM and NM keywords.
;                 Changed meaning of auto flag.
;               Version 1.1, 14/06/01
;                 Added MCP_PROCESS keyword.
;               Version 1.2, 04/07/01
;                 Added TEMP_PROCESS keyword.
;               Version 1.3, 07/07/01
;                 Added NPLOTS keyword.
;               Version 1.4, 19/07/01
;                 Added GSEOS keyword (unused at present)..
;               Version 1.5, 11/10/01
;                 Added 4SC printout
;               Version 1.6, 21/01/02
;                 Added REALTM, SURVEY keywords, removed BATCH_IF keyword
;               Version 1.7, 07/02/02
;                 Added check to see if 4SC plot exists before performing.
;                 Moved some printout.
;               Version 1.8, 14/02/02
;                 Put back BATCH_IF keyword for compatability with previous versions.
;               Version 1.9, 15/02/02
;                 Changed arguments to rcs_4sc_survey.
;                 Added RANGE, ERANGE keywords.
;               Version 2.0, 14/05/02
;                 Added PA, LINE and OCC 4SC survey plots.
;               Version 2.1, 16/05/02
;                 Added PEDESTAL PROCESSING.
;                 Renamed rcs_mcp and rcs_temperature routines.
;               Version 2.2, 20/06/02
;                 Corrected bug in which ERANGE setup incorrectly for OCCUPANCy plots.
;                 Moved defaults to subroutines.
;               Version 2.3, 04/07/02
;                 Added SCI data source.
;               Version 2.4, 10/07/02
;                 Modified ies_ncolors.
;                 Removed ids from rcs_batch_if argument list
;               Version 2.5, 25/11/02
;                 Added some printout.
;               Version 2.6, 31/12/02
;                 Added some printout.
;               Version 2.7, 17/01/03
;                 Changed set up of days and 4SC processing.
;               Version 2.8, 28/02/03
;                 Added SW_VERSION keyword.
;               Version 2.9, 30/08/03
;                 Changed to RCS environment variables for PNG and GIF.
;               Version 3.0, 03/10/03
;                 Added WEB keyword.
;               Version 3.1, 09/10/03
;                 Used ies get dir.
;               Version 3.2, 07/11/03
;                 Moved responsibility for creating sub-directories to lower level routines.
;                 Changed rcs batch output dir argument list.
;               Version 3.3, 16/12/03
;                 Added check for each SC that days found before trying to process
;
; Version     :	Version 3.3, 16/12/03
;-
;**********************************************************

PRO rcs_batch, DDS=dds, DAT=dat, SCI=sci, GSEOS=gseos, $
               TSTART=tstart, TSTOP=tstop, PROCESS_IF=process_if, $
               REPROCESS_IF=reprocess_if, IFCFG=ifcfg, FORCE=force, SILENT=silent, SC1=sc1, SC2=sc2, SC3=sc3, SC4=sc4, $
               POLAR_4SC=polar_4sc, IES_4SC=ies_4sc, IIMS_4SC=iims_4sc, PA_4SC=pa_4sc, LINE_4SC=line_4sc, OCC_4SC=occ_4sc, $
               HK=hk, NM=nm, BM=bm, NPLOTS=nplots, SOURCE=source, SW_VERSION=sw_version, $
               REALTM=realtm, SURVEY=survey, BATCH_IF=batch_if, $
               MCP_PROCESS=mcp_process, TEMP_PROCESS=temp_process, PED_PROCESS=ped_process

  ; initialize color table

  dummy = ies_ncolors(/SET, PRIVATE=39)

  ; set up widget ids
  ; nb type defines source

  IF KEYWORD_SET(dat) THEN BEGIN
    ids = {base:0L, text:0L, sc:0L, hk:0L, block:0L, start:0L, spacecraft:0L, type:'DAT', $
           in:REPLICATE({FILE, type:'EDB', file:0L, bytes:0L, time:0L},2), $
           skipid:0L, skiptime:0L, showedb:0L, time:0L, instrument:0L, $
           showif:0L, info:0L, auto:0L, datadir:0L}
    ids.in[1].type = 'HK'
  ENDIF ELSE IF KEYWORD_SET(sci) THEN BEGIN
    ids = {base:0L, text:0L, sc:0L, hk:0L, block:0L, start:0L, spacecraft:0L, type:'SCI', $
           skipid:0L, skiptime:0L, showedb:0L, time:0L, instrument:0L, $
           showif:0L, info:0L, auto:0L, datadir:0L}
  ENDIF ELSE BEGIN
    ids = {base:0L, text:0L, sc:0L, hk:0L, block:0L, start:0L, spacecraft:0L, type:'DDS', $
           in:REPLICATE({FILE, type:'NM', file:0L, bytes:0L, time:0L},3), $
           skipid:0L, skiptime:0L, showedb:0L, time:0L, instrument:0L, $
           showif:0L, info:0L, auto:0L, datadir:0L}
    ids.in[1].type = 'BM'
    ids.in[2].type = 'HK'
  ENDELSE

  rcs_set_ids, ids

  ; set up common block

  rcs_initialize_common, ids

  ; set auto mode
  ; auto : INT indicating BATCH IF mode.
  ;   0 => non-batch mode
  ;   1 => generate realtime plots
  ;   2 => generate single SC survey plots
  ;   3 => generate both plots
  ;   4 => generate neither plots

  IF KEYWORD_SET(batch_if) THEN BEGIN

    rcs_set_auto, batch_if

  ENDIF ELSE BEGIN

    auto = 0

    IF KEYWORD_SET(realtm) THEN auto = auto OR 1

    IF KEYWORD_SET(survey) THEN auto = auto OR 2

    IF NOT KEYWORD_SET(auto) THEN auto = 4

    rcs_set_auto, auto

  ENDELSE

  ; set silent mode

  IF KEYWORD_SET(silent) THEN rcs_set_silent, 1

  ; set force

  IF KEYWORD_SET(force) THEN rcs_set_force, 1

  ; set nplots

  IF KEYWORD_SET(nplots) THEN rcs_set_nplots, nplots

  ; set source

  IF datatype(source) EQ 'STR' THEN rcs_set_source, source

  ; set sw version

  IF datatype(sw_version) EQ 'STR' THEN rcs_set_sw_version, sw_version

  rcs_text, 'RCS BATCH PROCESS STARTED', /OVERRIDE

  ; check if IF CFG file

  IF KEYWORD_SET(ifcfg) THEN rcs_restore_use, FILE=ifcfg

  ; check which S/C to process

  isc = -1

  IF KEYWORD_SET(SC1) THEN isc = [isc,0]

  IF KEYWORD_SET(SC2) THEN isc = [isc,1]

  IF KEYWORD_SET(SC3) THEN isc = [isc,2]

  IF KEYWORD_SET(SC4) THEN isc = [isc,3]

  IF N_ELEMENTS(isc) EQ 1 THEN isc = INDGEN(4) ELSE isc = isc[1:*]

  ; loop over S/C

  sc = ['1','2','3','4']

  FOR k = 0, N_ELEMENTS(isc)-1 DO BEGIN

    ; reset files, common blocks and display

    rcs_reset, ids, /FILES

    rcs_text, 'RESET', /OVERRIDE

    ; check if processing directly from IF files

    IF KEYWORD_SET(process_if) OR KEYWORD_SET(reprocess_if) THEN BEGIN

      ; NB even if do not produce any plots still need to do this to get days

      rcs_text, 'START PROCESSING SURVEY PLOTS FOR S/C '+sc[isc[k]], /OVERRIDE

      days = rcs_add_days( sc[isc[k]], TSTART=tstart, TSTOP=tstop)

      rcs_batch_if, sc[isc[k]], days, REPROCESS_IF=reprocess_if, /WEB

    ENDIF ELSE IF KEYWORD_SET(mcp_process) THEN BEGIN

      rcs_text, 'START MCP PROCESSING S/C '+sc[isc[k]], /OVERRIDE

      days = rcs_add_days( sc[isc[k]], TSTART=tstart, TSTOP=tstop)

      rcs_batch_mcp, ids, sc[isc[k]], days, MCP_PROCESS=mcp_process

    ENDIF ELSE IF KEYWORD_SET(temp_process) THEN BEGIN

      rcs_text, 'START TEMPERATURE PROCESSING S/C '+sc[isc[k]], /OVERRIDE

      days = rcs_add_days( sc[isc[k]], TSTART=tstart, TSTOP=tstop)

      rcs_batch_temperature, ids, sc[isc[k]], days, TEMP_PROCESS=temp_process

    ENDIF ELSE IF KEYWORD_SET(ped_process) THEN BEGIN

      rcs_text, 'START PEDESTAL DATA PROCESSING S/C '+sc[isc[k]], /OVERRIDE

      days =  rcs_add_days( sc[isc[k]], TSTART=tstart, TSTOP=tstop)

      rcs_batch_pedestal, ids, sc[isc[k]], days, PED_PROCESS=ped_process

    ENDIF ELSE IF ids.type EQ 'SCI' THEN BEGIN

      ; NB even if do not produce any plots still need to do this to get days

      rcs_text, 'START PROCESSING SCI DATA FOR S/C '+sc[isc[k]], /OVERRIDE

      days = rcs_add_days( sc[isc[k]], TSTART=tstart, TSTOP=tstop, /SCI)

      rcs_batch_sci, sc[isc[k]], days

      rcs_text, 'START PROCESSING SURVEY PLOTS FOR S/C '+sc[isc[k]], /OVERRIDE

      rcs_batch_if, sc[isc[k]], days, /WEB

    ENDIF ELSE BEGIN

      ; get file names for processing

      IF ids.type EQ 'DDS' THEN BEGIN

        days = rcs_add_cd( ids, SC=sc[isc[k]], TSTART=tstart, TSTOP=tstop, HK=hk, NM=nm, BM=bm)

      ENDIF ELSE BEGIN

        days = rcs_add_dat( ids, SC=sc[isc[k]], TSTART=tstart, TSTOP=tstop, HK=hk, NM=nm, BM=bm)

      ENDELSE

      ; check if any data for this S/C

      IF KEYWORD_SET(days) THEN BEGIN

        rcs_text, 'START PROCESSING S/C '+sc[isc[k]], /OVERRIDE

        ; loop while data

        ok = 1

        WHILE ok DO BEGIN

          ; read more data

          ok = rcs_get_data(ids,/WEB)

          ; process all stored data

          rcs_get_sc_edbs, ids

          rcs_get_hk_edbs, ids

          rcs_extract_state, ids

          rcs_in_state, ids

        ENDWHILE

      ENDIF

    ENDELSE

    IF KEYWORD_SET(days) THEN IF N_ELEMENTS(all_days) EQ 0 THEN all_days = days ELSE all_days = [all_days, days]

  ENDFOR

  ; get unique days

  IF N_ELEMENTS(all_days) EQ 0 THEN RETURN

  list = rcs_sort_days(all_days)

  all_days = all_days[list]

  ; set up type of output required

  IF rcs_version() THEN BEGIN

    root = 'PNG'

    png = 1

  ENDIF ELSE BEGIN

    root = 'GIF'

    gif = 1

  ENDELSE

  ; override for WEB output

  root = 'WEB'

  ; set up 4SC survey plots

  types = [-1]
  cfgs = ''

  IF KEYWORD_SET(ies_4sc)   THEN types = [types,0]
  IF KEYWORD_SET(iims_4sc)  THEN types = [types,1]
  IF KEYWORD_SET(polar_4sc) THEN types = [types,2]
  IF KEYWORD_SET(pa_4sc)    THEN types = [types,3]
  IF KEYWORD_SET(line_4sc)  THEN types = [types,4]
  IF KEYWORD_SET(occ_4sc)   THEN types = [types,5]

  IF KEYWORD_SET(ies_4sc)   THEN cfgs = [cfgs,'survey_ies_4sc.cfg']
  IF KEYWORD_SET(iims_4sc)  THEN cfgs = [cfgs,'survey_iims_4sc.cfg']
  IF KEYWORD_SET(polar_4sc) THEN cfgs = [cfgs,'survey_polar_4sc.cfg']
  IF KEYWORD_SET(pa_4sc)    THEN cfgs = [cfgs,'survey_pa_4sc.cfg']
  IF KEYWORD_SET(line_4sc)  THEN cfgs = [cfgs,'survey_line_4sc.cfg']
  IF KEYWORD_SET(occ_4sc)   THEN cfgs = [cfgs,'survey_occ_4sc.cfg']

  IF N_ELEMENTS(types) GT 1 THEN BEGIN

    types = types[1:*]
    cfgs = cfgs[1:*]

  ENDIF ELSE RETURN

  ; do 4 SC survey plots

  rcs_text, 'START PROCESSING 4 SC DATA', /OVERRIDE

  FOR k = 0, N_ELEMENTS(all_days)-1 DO BEGIN

    ; do standard survey plots
    ; fool routine into getting names right

    FOR t = 0, N_ELEMENTS(types)-1 DO BEGIN

      FOR q = 0, rcs_get_nplots() DO BEGIN

        subdir = rcs_batch_output_dir(root, '', cfgs[t], rcs_strtrim(q))

        IF NOT KEYWORD_SET(subdir) THEN RETURN

        IF KEYWORD_SET(force) OR KEYWORD_SET(process_if) OR $
          NOT rcs_batch_output_file(all_days[k], subdir, PNG=png, GIF=gif) THEN $
            rcs_4sc_survey, DAY=all_days[k], PNG=png, GIF=gif, TYPE=types[t], NPLOT=q

      ENDFOR

    ENDFOR

  ENDFOR

END
