;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_MAG_SC
;
; Purpose     :	Works out the magnetic field in the SC reference frame for the M values
;               and M signs over a spin.
;               The m-value is the polar direction of the perpendicular to the B-field
;               expressed as 0-15 with 0 along the SC spin_axis.
;               At 90 degrees to this is the direction closest to the B field direction.
;               The sense of this is determined from the M-sign.
;
; Use         : < mag = RCS_MAG_SC(mvals) >
;
; Inputs      : mvals : BYTARR containing the [8]BYTE m values determining the detectors
;                       for each sector, [2]BYTE msigns determining B-field parallel or
;                       anti-parallel.
;
; Opt. Inputs : None.
;
; Outputs     : mag : [2]FLTARR giving magnetic field azimuth and polar angles.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 27/02/02
;
; Modified    : Version 0.1, 06/10/03
;                 Corrected bug in calculation of polar angle when msign=1
;                 Had : b = [((ms+0.5+msigns[ms]*8) MOD 16)/16.0*2*!PI,((mp[ms]+0.5+msigns[ms]*16) MOD 16)/16.*!PI]
;                 but this does not do the rotation in polar angle correctly
;
; Version     :	Version 0.1, 06/10/03
;-
;**********************************************************

FUNCTION rcs_mag_sc, mvals

  ; get m values for each sector
  ; m = 0-15

  m = rcs_decode_m(mvals, /MVAL)

  ; get msigns for each sector

  msigns = rcs_decode_msigns(mvals)

  ; find the sector containing the B field
  ; works out sector containing the minimum m value
  ; this should be where the perp B field is in the 1st quadrant ie ms = 0-7
  ; but can't rely on it

  ms = rcs_min_sector(m)

  ; get parallel direction 0-15

  ; if B perp is in 1st quadrant and msign is zero then B parallel lies in 2nd quadrant of this sector
  ; if B perp is in 1st quadrant and msign is one  then B parallel lies in 1st quadrant of opposite sector
  ; if B perp is in 2nd quadrant and msign is zero then B parallel lies in 1st quadrant of this sector
  ; if B perp is in 2nd quadrant and msign is one  then B parallel lies in 3rd quadrant of opposite sector

  ; this works for B perp in 1st or 2nd quadrant
  ; NB 0.5 contribution puts the m value at the centre of the polar bin

  mp = (m[FIX(ms)] +0.5 + 8) MOD 16

  ; rotate through 180 degrees if msign set

  IF msigns[FIX(ms)] THEN mp = 16 - mp

  ; get B field direction
  ; NB 0.5 contribution puts the m value and sector measurements at the centre of the polar and sector bins
  ; field direction given as [azimuth,polar]

  b = [((ms+0.5+msigns[ms]*8) MOD 16)/16.0*2*!PI,mp/16.*!PI]

  IF 0 THEN BEGIN

    PRINT, m, FORMAT='(16I5)'
    PRINT, msigns, FORMAT='(16I5)'
    PRINT, mp, FORMAT='(16I5)'

    fmp = CURVEFIT(INDGEN(16),mp,REPLICATE(1,16),b,FUNCTION_NAME='rcs_find_mp',/NODERIVATIVE)

  ENDIF

  RETURN, b

END
