;+
; Project     :	CLUSTER - PEACE
;
; Name        :	RCS_TIME_CENTROID
;
; Purpose     :	Produces time centroid of data
;
; Explanation :	Produces time centroid of data. Centroids over fixed time spans.
;               If no data in a time span then time and centroid are zero.
;
; Use         : < rcs_time_centroid >
;
; Inputs      : time_start : DOUBLE TAI start time of ROI
;               time_end   : DOUBLE TAI stop time  of ROI
;               time       : DBLARR[nsamples] TAI start times of sample.
;               data       : FLTARR[ndetectors,nbands,nsamples] data
;
; Opt. Inputs : None.
;
; Outputs     : atime  : DBLARR[] TAI new sample start times.
;               adata  : FLTARR[] averaged sample data.
;
; Opt. Outputs:	None.
;
; Keywords    : PERIOD : LONG time span in seconds. Only truncates to one day.
;
; Written     :	Version 0.0, 06/02/02
;
; Modified    : 
;;
; Version     : Version 0.0, 06/02/02
;-
;**********************************************************

PRO rcs_time_centroid, time_start, time_end, time, data, atime, adata, PERIOD=period, POINTS=points

  IF NOT KEYWORD_SET(period) THEN period = 60L

  ; assume period is less than or equal to one day
  ; to avoid problems with leap seconds use MJD format

  tmjd = tai2utc(time_start)

  period_ms = period*1000L

  tmjd.time = (tmjd.time/period_ms)*period_ms

  tstart = utc2tai(tmjd)

  ; set up new samples

  anp = LONG((time_end-tstart)/period+0.99999)

  IF anp EQ 0 THEN RETURN

  atime = tstart+DINDGEN(anp)*period

  IF NOT KEYWORD_SET(data) THEN BEGIN

    adata = 0

    RETURN

  ENDIF

  adata = FLTARR( (SIZE(data))[1], (SIZE(data))[2], anp )

  points = LONARR(anp)

  ; get new samples

  k = 0L

  kstart = 0

  FOR n = 0L, anp-1 DO BEGIN

    WHILE (time[k] LT atime[n]+period) AND (k LT N_ELEMENTS(time)-1) DO k = k + 1

    kpoints = k - kstart

    IF kpoints EQ 1 THEN BEGIN

      adata[*,*,n] = data[*,*,kstart]

    ENDIF ELSE IF kpoints GT 1 THEN BEGIN

      sum = 0
      FOR i = kstart, k-1 DO BEGIN
      
        sum = sum + data[*,*,i]*time[i]

      ENDFOR

      adata[*,*,n] = sum/TOTAL(data[*,*,kstart:k-1],3)

    ENDIF

    points[n] = kpoints

    kstart = k

  ENDFOR

END
