;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_DO_CAL
;
; Purpose     :	Applies calibration factors to data (but not cleaning).
;
; Explanation :	Applies geometric factor, efficiency, energy bin width and sample time calibrations using info stored in
;               IF data set header.
;
; Use         : < rcs_do_cal, header, data, output_info >
;
; Inputs      : header      : STRUCTURE containing IF data set info
;               data        : STRUCTURE array containijg IF data
;               output_info : STRUCTURE containing display info
;
; Opt. Inputs : None.
;
; Outputs     : output_array : FLTARR(nsectors,ndetectors,newbands,npoints) calibrated data
;
; Opt. Outputs:	None.
;
; Keywords    : NOCAL  : FLAG indicating not to apply calibration factors
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 09/04/02
;                 Coped with case where only single point.
;               Version 0.2, 14/04/03
;                 Changed argument list of ies_rat.
;               Version 0.3, 24/10/03
;                 Used ies lut list.
;               Version 0.4, 20/11/03
;                 Used routine ies bins to kev..
;                 Renamed routine rcs docal to rcs do cal.
;
; Version     :	Version 0.4, 20/11/03
;-
;**********************************************************

FUNCTION rcs_do_cal, header, data, output_info, NOCAL=nocal

  ; reform output array
  ; NB brackets needed if only single point ?!

  output_array = REFORM ( [data.data], header.nsectors, header.ndetectors, header.nbands, N_ELEMENTS(data))

  IF KEYWORD_SET(nocal) THEN RETURN, output_array

  ; get list of lut runs

  lut_list = ies_lut_list(data.lut)

  ; take into account channel sizes
  ; convert from counts per channel to count per keV

  IF output_info.div AND NOT header.div THEN $
    ies_div, lut_list, header.nbands, header.ndetectors, $
          ies_bins_to_kev(header.energy_cal, header.ch_positions), output_array

  ; apply rate calibration i.e convert from counts to counts/sec
  ; store rates as count rate per nsectors as for POLAR
  ; to convert from count rate / nsectors to count rate multiply by the no of sectors
  ; ie accumulation time per sector is spin_time / nsectors

  IF output_info.rat  AND NOT header.rat THEN ies_rat, data, header.nsectors, header.nbands, header.ndetectors, output_array, REVERSE=reverse

  ; apply count calibration i.e convert from counts to counts/sr/cm2

  IF output_info.cal  AND NOT header.cal THEN ies_cal, lut_list, header.count_cal, output_array

  IF output_info.eff THEN ies_efficiency, lut_list, header.energy_efficiency, output_array

  ; return array

  RETURN, output_array

END
