;+
; Project     :	CLUSTER - PEACE
;
; Name        :	RCS_MK_IMAGE
;
; Purpose     :	Produces rebinned 2-D image of counts for y vs x.
;
; Explanation :	Associated with each column of the input image is a LUT number which defines the
;               bin positions of the y samples. This routine maps the input image into a new image
;               size in both x and y axes optionally taking into account the different column to
;               column y mapping i.e the different y channel sizes.
;               It also allows an option to shift each column to match pedestal positions.
;
; Use         : < rcs_mk_image, y, x, lut_list, ch_positions, nchannels, ylo, yhi, xsize, ysize, $
;                               SHOWCH=showch, SHFTCH=shftch, LOG=log >
;
; Inputs      : y            : FLTARR(nchannels,npoints) data values at each sample point for each channel
;               x            : DBLARR(npoints) times for each sample in TAI format
;               lut_list     : STRUCTURE ARRAY congtining LUT run info.
;               ch_positions : FLTARR(2,nchannels,1,nluts) channel bin positions for each LUT.
;               nchannels    : LONG no. of y channels
;               ylo          : LONG low y axis value
;               yhi          : LONG high y axis value
;               xsize        : LONG x dimension of output array
;               ysize        : LONG y dimension of output array
;
; Opt. Inputs : None.
;
; Outputs     : y            : FLTARR(ysize,xsize) image
;
; Opt. Outputs:	None.
;
; Keywords    :/SHOWCH   : indicates whether to show real channel sizes or not
;              /SHFTCH   : indicates whether to shift data to match pedestal positions.
;                          Only available if showch is also set.
;              /LOG      : indicates logarithmic y axis display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/6/95
;
; Modified    :	Version 0.1, MKC, 19/9/96
;                 Improved pedestal shift algorithm.
;                 Added DIV keyword.
;                 Added INTERP keyword.
;                 Shift pedestal to zero.
;                 Added ies rebin y.
;               Version 0.2, 11/11/96
;                 Removed DIV, new argument list for rebin_y
;               Version 0.3, 1/4/97
;                 Added LOG keyword.
;                 Changed yp to channel boundaries rather than channel centres.
;               Version 0.4, 28/4/97
;                 Corrected routine for case where more than 32767 data points.
;               Version 0.5, 2/3/98
;                 Removed interpolation option. Removed minimum argument.
;               Version 0.6, 7/12/00
;                 Fixed bug in ensuring device_y in range.
;               Version 0.7, 20/07/01
;                 Used ies runs.
;               Version 0.8, 24/10/03
;                 Changed to lut list argument.
;
; Version     :	Version 0.8, 24/10/03
;-
;**********************************************************

PRO rcs_mk_image, y, x, lut_list, output_chans, conv, output_info

  ; where one device pixel covers a number of samples would like to average samples together
  ; this requires keeping track of no. of samples at each pixel and total counts
  ; i.e 2 LONGs per pixel. This may or may not be too great an overhead.

  ; x = DBLARR[2,npoints]

  npoints = (SIZE(x))[2]

  ; y = DBLARR[nchannels,npoints]

  nchannels = (SIZE(y))[1]

  ; calculate x positions of input data in device pixels

  device_x = LONG ( ((x-conv.xrange[0]) * conv.xsize)/ (conv.xrange[1]-conv.xrange[0]) )

  ; truncate if out of range

  device_x = (conv.xsize-1) < device_x > 0

  ; loop through runs of each lut

  FOR l = 0, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut

    lut = lut_list[l].lut

    ; output_chans is [2,nchannels,ndetectors=1,nluts]

    IF KEYWORD_SET(output_info.showch) THEN BEGIN
      chans = output_chans[*,*,0,lut]
    ENDIF ELSE BEGIN
      chans = INTARR(2,nchannels)
      chans[0,*] = INDGEN(nchannels)
      chans[1,*] = 1+INDGEN(nchannels)
    ENDELSE

    ; get device pixels for each channel
    ; NB if YLOG then !Y.CRANGE contains ALOG10(yrange)

    IF KEYWORD_SET(output_info.elog) THEN BEGIN

      device_y = LONG ( ((ALOG10(chans>1e-20)-conv.yrange[0]) * conv.ysize)/ (conv.yrange[1]-conv.yrange[0]) )

    ENDIF ELSE BEGIN

      device_y = LONG ( ((chans-conv.yrange[0]) * conv.ysize)/ (conv.yrange[1]-conv.yrange[0]) )

    ENDELSE

    ; truncate if out of range

    device_y = (conv.ysize-1) < device_y > 0

    ; for every device pixel find corresponding channel : -1 if none
    ; should not be any overlapping of channels so unique channel no for each pixel

    mapc = REPLICATE(-1,conv.ysize)

    FOR c = 0, nchannels - 1 DO FOR yy = device_y[0,c], device_y[1,c] DO mapc[yy] = c

    list = WHERE(mapc GE 0, count)

    IF count GT 0 THEN BEGIN

      ; get channel nos for those pixels associated with a channel

      mapc = mapc[list]

      ; loop over run of samples with the same lut

      FOR k = lut_list[l].st, lut_list[l].en DO BEGIN

        ; get freq image

        freq = conv.freq[device_x[0,k]:device_x[1,k]]
        ;freq = conv.freq[device_x[0,k]:device_x[1,k],list]

        ; increment frequency

        new_freq = freq+1

        ; get ratio

        freq_ratio = [(freq/new_freq)]#REPLICATE(1,count)
        ;freq_ratio = freq/new_freq

        ; extract y samples for each pixel and form
        ; 2D array to be added to image

        yk = [(1/new_freq)]#y[mapc,k]
        ;yk = (REPLICATE(1,device_x[1,k]-device_x[0,k]+1)#y[mapc,k])/new_freq

        ; add sample to image

        conv.data[device_x[0,k]:device_x[1,k],list] = conv.data[device_x[0,k]:device_x[1,k],list]*freq_ratio + yk

        ; increment frequency

        conv.freq[device_x[0,k]:device_x[1,k]] = new_freq
        ;conv.freq[device_x[0,k]:device_x[1,k],list] = new_freq

      ENDFOR

    ENDIF

  ENDFOR

  rcs_display_image, output_info, conv, device_x[0,0], device_x[1,npoints-1], device_y[0,0], device_y[1,nchannels-1]

END
