;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PANEL_DRAW
;
; Purpose     :	Sets up axes in display panels.
;
; Explanation :	Sets up axes in display panels.
;
; Use         : < rcs_panel_draw, panel_ids, output_info, overlap >
;
; Inputs      : panel_ids   : STRUCTURE containing widget ids of display panels
;               output_info : STRUCTURE containing display information
;               overlap     : STRUCTURE giving amout of overlap with previous display
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs:	None.
;
; Keywords    : INFO : STRUCTURE containing non-widget output info
;               NO_OVERLAP  : FLAG indicates should not try to use overlap
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 13/12/00
;                 Added NO_OVERLAP keyword.
;                 Fixed bug in PAD plots which fixed cttile on counts.
;               Version 0.2, 6/2/01
;                 Made TMP parameters controllable.
;                 Added ESPCT plot.
;                 Changed centroid to pedestal plot.
;               Version 0.3, 15/05/01
;                 Added CLEANBM2
;               Version 0.4, 25/05/01
;                 Added HKSUN and HKRAT.
;                 Added NM/BM to HKFLAG plot.
;                 Removed STA and STO from SGL0/1 plot.
;               Version 0.5, 31/05/01
;                 Added STABM and STOBM.
;               Version 0.6, 09/09/01
;                 Modified PEDESTAL plot.
;               Version 0.7, 08/02/02
;                 Added EPAD2.
;               Version 0.8, 16/05/02
;                 Modified PEDESTAL plot.
;
; Version     :	Version 0.8, 16/05/02
;-
;**********************************************************

PRO rcs_panel_draw, panel_ids, output_info, overlap, INFO=info, NO_OVERLAP=no_overlap

  ; check if non-widget output wanted

  IF KEYWORD_SET(info) THEN BEGIN

    ysiz = info.ysiz/(6 > output_info.noutputs)

    !P.REGION = [ 0.0, info.ytop, 1.0, info.ytop+ysiz]

  ENDIF

  ; check if margins defined

  xmargin = [11,12]

  ymargin = [1.5,0.6]

  ; set up names

  epad_names = ['Lo0', 'Lo1', 'Lo2', 'Hi0', 'Hi1', 'Hi2']

  ehk_names = rcs_strtrim(INDGEN(N_ELEMENTS(output_info.use_ehk)))

  e3dd_names = rcs_strtrim(INDGEN(N_ELEMENTS(output_info.use_dets)))

  i3dd_names = rcs_strtrim(INDGEN(N_ELEMENTS(output_info.use_i3dd)))

  i3ddm_names = ['H','He','CNO']

  ipad_names = ['Lo0', 'Lo1', 'Lo2', 'Hi0', 'Hi1', 'Hi2']

  sglbm_names = ['ENY','TCR','TAC']

  ; do ESPCT panel

  IF output_info.espct NE 0 THEN rcs_panel_draw_spanel, panel_ids.espct, output_info, overlap, xmargin, ymargin, $
    STITLE='ESPCT', CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, NO_OVERLAP=no_overlap

  ; do E3DD panels

  detector_list = WHERE(output_info.use_dets, count)

  FOR k = 0, output_info.e3dd-1 DO rcs_panel_draw_spanel, panel_ids.e3dds[k], output_info, overlap, xmargin, ymargin, $
    STITLE='E3DD:'+e3dd_names[detector_list[k]], CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, NO_OVERLAP=no_overlap

  ; do E3DDBM panels

  detector_list = WHERE(output_info.use_bmdets, count)

  FOR k = 0, output_info.e3ddbm-1 DO rcs_panel_draw_spanel, panel_ids.e3ddbms[k], output_info, overlap, xmargin, ymargin, $
    STITLE='E3DDBM:'+e3dd_names[detector_list[k]], CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, SECTORS=output_info.sectors, NO_OVERLAP=no_overlap

  ; do CLEAN panels

  detector_list = WHERE(output_info.use_cleans, count)

  FOR k = 0, output_info.clean-1 DO rcs_panel_draw_spanel, panel_ids.cleans[k], output_info, overlap, xmargin, ymargin, $
    STITLE='CLEAN:'+e3dd_names[detector_list[k]], CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, NO_OVERLAP=no_overlap

  ; do CLEANBM panels

  detector_list = WHERE(output_info.use_bmcleans, count)

  FOR k = 0, output_info.cleanbm-1 DO rcs_panel_draw_spanel, panel_ids.cleanbms[k], output_info, overlap, xmargin, ymargin, $
    STITLE='CLEANBM:'+e3dd_names[detector_list[k]], CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, SECTORS=output_info.sectors, NO_OVERLAP=no_overlap

  ; do CLEANBM2 panels

  detector_list = WHERE(output_info.use_bm2cleans, count)

  FOR k = 0, output_info.cleanbm2-1 DO rcs_panel_draw_spanel, panel_ids.cleanbm2s[k], output_info, overlap, xmargin, ymargin, $
    STITLE='CLEANBM2:'+e3dd_names[detector_list[k]], CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, SECTORS=output_info.sectors, NO_OVERLAP=no_overlap

  ; do PEDESTAL panels

  detector_list = WHERE(output_info.use_pedestals, count)

  FOR k = 0, output_info.pedestal-1 DO BEGIN

    IF output_info.noutputs EQ output_info.pedestal OR $
       output_info.noutputs EQ output_info.pedestal+output_info.luts THEN BEGIN

      pmin = output_info.minimum
      pmax = output_info.maximum

    ENDIF ELSE BEGIN

      pmin = -10
      pmax = 20

   ENDELSE

    rcs_panel_draw_lpanel, panel_ids.pedestals[k], output_info, xmargin, ymargin, $
        STITLE='PED:'+e3dd_names[detector_list[k]], MINIMUM=pmin, MAXIMUM=pmax, YTITLE='PED. SHIFT', LOG=0, $
        SUBTITLES=['x-corr','centroid','-%occ','-%miss'], YSIZ=ysiz

   ENDFOR

  ; do EHIST panel

  IF output_info.ehist NE 0 THEN BEGIN

    IF output_info.noutputs EQ 1 THEN hist = 0 ELSE hist = 1

    rcs_panel_draw_spanel, panel_ids.ehist, output_info, overlap, xmargin, ymargin, HIST=hist, $
          STITLE='EHIST', CTITLE='Counts', YSIZ=ysiz, NO_OVERLAP=no_overlap

  ENDIF

  ; do FIT panel

  IF output_info.fit NE 0 THEN BEGIN

    rcs_panel_draw_lpanel, panel_ids.fit[0], output_info, xmargin, ymargin, $
      STITLE='PED FIT', MINIMUM=0, MAXIMUM=10, YTITLE='Ped. FWHM, POS/10', $
      LOG=0, SUBTITLES=['FWHM','POS/10'], YSIZ=ysiz

    rcs_panel_draw_lpanel, panel_ids.fit[1], output_info, xmargin, ymargin, $
      STITLE='PED SHF', MINIMUM=-15, MAXIMUM=15, YTITLE='Ped. shift', LOG=0, SUBTITLES=['SHFT'], YSIZ=ysiz

    rcs_panel_draw_lpanel, panel_ids.fit[2], output_info, xmargin, ymargin, $
      STITLE='PED FWR', MINIMUM=0, MAXIMUM=2, YTITLE='Ped. FWHM ratio', LOG=0, SUBTITLES=['FWR'], YSIZ=ysiz

  ENDIF

  ; do EHK panel

  detector_list = WHERE(output_info.use_ehk, count)

  FOR k = 0, output_info.ehk-1 DO rcs_panel_draw_lpanel, panel_ids.ehks[k], output_info, xmargin, ymargin, $
      STITLE='HK:'+ehk_names[detector_list[k]], SUBTITLES=['EHK0','EHK1','EHK2','EHK3'], $
      YTITLE='Counts', YSIZ=ysiz

  ; do EPAD panels

  detector_list = WHERE(output_info.use_epad, count)

  FOR k = 0, output_info.epad-1 DO rcs_panel_draw_spanel, panel_ids.epads[k], output_info, overlap, xmargin, ymargin, $
    /SECTORS, STITLE='EPAD:'+epad_names[detector_list[k]], /NO_OVERLAP, CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz

  ; do EPAD2 panels

  detector_list = WHERE(output_info.use_epad2, count)

  FOR k = 0, output_info.epad2-1 DO rcs_panel_draw_lpanel, panel_ids.epad2s[k], output_info, xmargin, ymargin, $
      STITLE='EPAD2:'+epad_names[detector_list[k]], SUBTITLES=['Dt0','Dt1','Dt2'], $
      YTITLE=ies_ctitle(0,output_info), YSIZ=ysiz

  ; do LUT panel

  IF output_info.luts NE 0 THEN rcs_panel_draw_lpanel, panel_ids.lut, output_info, xmargin, ymargin, STITLE='LUTS', $
    MINIMUM=-1.5, MAXIMUM=3.5, YTITLE='Integration mode', SUBTITLES=['NM', 'BM1', 'BM2', 'BM3', 'HIST', 'HK'], $
    YTICKV=[-1,0,1,2,3], LOG=0, YTICKNAME=['X', '2 us', '5 us', '15 us', '50 us'], YSIZ=ysiz

  ; do SCFLAG panel

  IF output_info.scflag NE 0 THEN rcs_panel_draw_lpanel, panel_ids.scflag, output_info, xmargin, ymargin, STITLE='SC-FLAG', $
    MINIMUM=-0.5, MAXIMUM=10.5, YTITLE='', SUBTITLES=['Unset', 'Set'], YTICKV=INDGEN(11), LOG=0, $
    YTICKNAME=['OK','IFFT', 'EPP', 'RAM', 'CLASS', 'IES_IF', 'EPP', 'NM/BM', 'BM1/BM2','LUTX','SUBC'], YSIZ=ysiz

  ; do SUBC panel

  IF output_info.subc NE 0 THEN rcs_panel_draw_lpanel, panel_ids.subc, output_info, xmargin, ymargin, STITLE='SUBC', $
    MINIMUM=0, MAXIMUM=255, YTITLE='', LOG=0, YSIZ=ysiz

  ; do HKFLAG panel

  IF output_info.hkflag NE 0 THEN rcs_panel_draw_lpanel, panel_ids.hkflag, output_info, xmargin, ymargin, STITLE='HK-FLAG', $
    MINIMUM=-0.5, MAXIMUM=11.5, YTITLE='', $
    SUBTITLES=['Unset','Set','TRIGM','SENSM'], YTICKNAME=['E+T','E+(D+T)','E*D*T','E*T','E','T','Serial','Parallel','IFFT','DEADT','AUTO','NM/BM'], $
    YTICKV=INDGEN(12), LOG=0, YSIZ=ysiz

  ; do HKFCR panel

  IF output_info.hkfcr NE 0 THEN rcs_panel_draw_lpanel, panel_ids.hkfcr, output_info, xmargin, ymargin, STITLE='HKFCR', $
    MINIMUM=0, MAXIMUM=31, YTITLE='', LOG=0, YSIZ=ysiz

  ; do HK TEMPERATURE panel

  IF output_info.tmp NE 0 THEN BEGIN
    IF output_info.noutputs EQ 1 THEN $
      rcs_panel_draw_lpanel, panel_ids.tmp, output_info, xmargin, ymargin, $
        STITLE='TEMP.', YTITLE='TEMPERATURE [degc]', $
        SUBTITLES=['ERISTREF','ERIHKTRF'], YSIZ=ysiz $
    ELSE $
      rcs_panel_draw_lpanel, panel_ids.tmp, output_info, xmargin, ymargin, $
        STITLE='TEMP.', MINIMUM=-20, MAXIMUM=40, YTITLE='TEMPERATURE [degc]', LOG=0, $
        SUBTITLES=['ERISTREF','ERIHKTRF'], YSIZ=ysiz
  ENDIF

  ; do I3DD panels

  channel_list = WHERE(output_info.use_i3ddm, mcount)

  detector_list = WHERE(output_info.use_i3dd, count)

  FOR k = 0, output_info.i3dd-1 DO rcs_panel_draw_spanel, panel_ids.i3dds[k], output_info, overlap, xmargin, ymargin, $
    STITLE='I3DD:'+i3ddm_names[channel_list[k/count]]+':'+i3dd_names[detector_list[k MOD count]], $
    CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, SECTORS=output_info.sectors, NO_OVERLAP=no_overlap

  ; do IPAD panels

  detector_list = WHERE(output_info.use_ipad, count)

  FOR k = 0, output_info.ipad-1 DO rcs_panel_draw_spanel, panel_ids.ipads[k], output_info, overlap, xmargin, ymargin, /SECTORS, $
    STITLE='IPAD:'+ipad_names[detector_list[k]], /NO_OVERLAP, CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz

  ; do HSPCT panel

  IF output_info.hspct NE 0 THEN rcs_panel_draw_spanel, panel_ids.hspct, output_info, overlap, xmargin, ymargin, STITLE='HSPCT', $
    CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, NO_OVERLAP=no_overlap

  ; do ISPCT-He panel

  IF output_info.ispctHe NE 0 THEN rcs_panel_draw_spanel, panel_ids.ispctHe, output_info, overlap, xmargin, ymargin, $
    STITLE='ISPCT-He', CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, NO_OVERLAP=no_overlap

  ; do ISPCT-CNO panel

  IF output_info.ispctCNO NE 0 THEN rcs_panel_draw_spanel, panel_ids.ispctCNO, output_info, overlap, xmargin, ymargin, $
    STITLE='ISPCT-CNO', CTITLE=ies_ctitle(0,output_info), YSIZ=ysiz, NO_OVERLAP=no_overlap

  ; do sglbm panel

  IF output_info.sglbm NE 0 THEN BEGIN

    IF output_info.sectors THEN BEGIN

      FOR k = 0, output_info.sglbm-1 DO rcs_panel_draw_spanel, panel_ids.sglbm[k], output_info, overlap, xmargin, ymargin, $
        STITLE='SGLBM:'+sglbm_names[k], CTITLE='Counts', YSIZ=ysiz, SECTORS=output_info.sectors, NO_OVERLAP=no_overlap

    ENDIF ELSE BEGIN

      rcs_panel_draw_lpanel, panel_ids.sglbm[0], output_info, xmargin, ymargin, $
        STITLE='SGLBM', SUBTITLES=sglbm_names, YTITLE='Counts', YSIZ=ysiz

    ENDELSE

  ENDIF

  ; do stabm panel

  IF output_info.stabm NE 0 THEN BEGIN

    IF output_info.sectors THEN BEGIN

      rcs_panel_draw_spanel, panel_ids.stabm, output_info, overlap, xmargin, ymargin, $
        STITLE='STABM', CTITLE='Counts', YSIZ=ysiz, SECTORS=output_info.sectors, NO_OVERLAP=no_overlap

    ENDIF ELSE BEGIN

      rcs_panel_draw_lpanel, panel_ids.stabm, output_info, xmargin, ymargin, $
        STITLE='STABM', YTITLE='Counts', YSIZ=ysiz

    ENDELSE

  ENDIF

  ; do stobm panel

  IF output_info.stobm NE 0 THEN BEGIN

    IF output_info.sectors THEN BEGIN

      rcs_panel_draw_spanel, panel_ids.stobm, output_info, overlap, xmargin, ymargin, $
        STITLE='STOBM', CTITLE='Counts', YSIZ=ysiz, SECTORS=output_info.sectors, NO_OVERLAP=no_overlap

    ENDIF ELSE BEGIN

      rcs_panel_draw_lpanel, panel_ids.stobm, output_info, xmargin, ymargin, $
        STITLE='STOBM', YTITLE='Counts', YSIZ=ysiz

    ENDELSE

  ENDIF

  ; do sgl01 panel

  IF output_info.sgl01 NE 0 THEN rcs_panel_draw_lpanel, panel_ids.sgl01, output_info, xmargin, ymargin, $
    STITLE='SGL0/1', SUBTITLES=['ENY','TCR','TAC'], YTITLE='Counts', YSIZ=ysiz

  ; do sta panel

  IF output_info.sta NE 0 THEN rcs_panel_draw_lpanel, panel_ids.sta, output_info, xmargin, ymargin, $
    STITLE='STA', SUBTITLES=['STA0-7','STA8-15'], YTITLE='Counts', YSIZ=ysiz

  ; do sto panel

  IF output_info.sto NE 0 THEN rcs_panel_draw_lpanel, panel_ids.sto, output_info, xmargin, ymargin, $
    STITLE='STO', SUBTITLES=['STO0-7','STO8-15'], YTITLE='Counts', YSIZ=ysiz

  ; do sgl1 panel

  IF output_info.sgl1 NE 0 THEN rcs_panel_draw_lpanel, panel_ids.sgl1, output_info, xmargin, ymargin, $
    STITLE='SGL1', SUBTITLES=['ENY','TCR','TAC'], YTITLE='Counts', YSIZ=ysiz

  ; do sgl2 panel

  IF output_info.sgl2 NE 0 THEN rcs_panel_draw_lpanel, panel_ids.sgl2, output_info, xmargin, ymargin, $
    STITLE='SGL2', SUBTITLES=['EDI1','EDI2','EDI3','BDI1','BDI2','BDI3'], YTITLE='Counts', YSIZ=ysiz

  ; do sgl3 panel

  IF output_info.sgl3 NE 0 THEN rcs_panel_draw_lpanel, panel_ids.sgl3, output_info, xmargin, ymargin, $
    STITLE='SGL3', SUBTITLES=['OVF1','OVF2','OVF3','SDIR-S1','SDIR-S2','SDIR-S3','SDIR-3S','TAC-S1','TAC-S2','TAC-S3'], $
    YTITLE='Counts', YSIZ=ysiz

  ; do hksun panel

  IF output_info.hksun NE 0 THEN rcs_panel_draw_lpanel, panel_ids.hksun, output_info, xmargin, ymargin, $
    STITLE='HKSUN', SUBTITLES=['SPPOS','SPSEC'], YTITLE='POS', YSIZ=ysiz, MINIMUM=0, MAXIMUM=16, LOG=0

  ; do hkrat panel

  IF output_info.hkrat NE 0 THEN rcs_panel_draw_lpanel, panel_ids.hkrat, output_info, xmargin, ymargin, $
    STITLE='HKRAT', SUBTITLES=['STA','STO','ENY'], $
    YTITLE='Counts', YSIZ=ysiz

  ; do HV panel

  IF output_info.hv NE 0 THEN BEGIN

    IF output_info.noutputs EQ 1 THEN $
      rcs_panel_draw_lpanel, panel_ids.hv, output_info, xmargin, ymargin, $
        STITLE='HV', YTITLE='HV [kV]', SUBTITLES=['ERISTAHV','ERISTOHV','ERIDEFHV'], YSIZ=ysiz $
    ELSE $
      rcs_panel_draw_lpanel, panel_ids.hv, output_info, xmargin, ymargin, $
        STITLE='HV', YTITLE='HV [kV]', SUBTITLES=['ERISTAHV','ERISTOHV','ERIDEFHV'], $
        MINIMUM=0, MAXIMUM=11, LOG=0, YSIZ=ysiz

  ENDIF

  IF output_info.m NE 0 THEN BEGIN

    rcs_panel_draw_lpanel, panel_ids.m[0], output_info, xmargin, ymargin, $
      STITLE='M', MINIMUM=0, MAXIMUM=15, YTITLE='M DIRECTION', LOG=0, YSIZ=ysiz

    rcs_panel_draw_lpanel, panel_ids.m[1], output_info, xmargin, ymargin, $
      STITLE='M SIGNS', MINIMUM=-2, MAXIMUM=2, YTITLE='M SIGNS', LOG=0, YSIZ=ysiz

  ENDIF

END
