;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PANEL_DRAW_SPANEL
;
; Purpose     :	Does single spectral plot of counts per channel vs time.
;
; Explanation :	Sets up axes, draws color bar.
;               Saves coordinate conversion.
;               If overlap specified then shifts existing stored image
;               to avoid overhead of redrawing whole image.
;
; Use         : < rcs_panel_draw_spanel, widget_id, output_info, overlap, xmargin, ymargin, $
;                           SECTORS=sectors, HIST=hist, STITLE=stitle, NO_OVERLAP=no_overlap >
;
; Inputs      : widget_id    : LONG widget id of draw window
;               output_info  : STRUCTURE containing display information
;               overlap      : STRUCTURE containing time limits of previous display
;               xmargin      : FLTARR(2) margins on x axis
;               ymargin      : FLTARR(2) margins on y axis
;
; Opt. Inputs : None.
;
; Outputs     : None..
;
; Opt. Outputs:	None.
;
; Keywords    : STITLE      : STRING title on rhs vertical
;               SECTORS     : FLAG indicates sector plot required
;               HIST        : FLAG indicates plotting IES histogram
;               NO_OVERLAP  : FLAG indicates should not try to use overlap
;               YSIZ        : FLOAT size of panel in plot window. If set indicates non-widget output.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, MC, 10/12/00
;                 Removed time range processing to rcs_panel_draw.
;                 Changed so that does not allow option of not using conv.data and freq for non-overlap spectrograms,
;                 Changed so that alters output_info before drawing axes.
;                 Change start_pix test to take into account arithmetical error.
;               Version 0.2, 30/05/01
;                 Added COMMON block for non-widget plots.
;               Version 0.3, 09/07/02
;                 Removed output_info.ncolors tag and ncolors
;
; Version     :	Version 0.3, 09/07/02
;-
;**********************************************************

PRO rcs_panel_draw_spanel, widget_id, output_info, overlap, xmargin, ymargin, $
                           SECTORS=sectors, HIST=hist, STITLE=stitle, NO_OVERLAP=no_overlap, CTITLE=ctitle, YSIZ=ysiz

  IF NOT KEYWORD_SET(ysiz) THEN BEGIN

    WIDGET_CONTROL, widget_id, GET_VALUE=win

    WSET, win

    !P.CHARSIZE = 1.0

    !P.REGION = [ 0.0, 0.0, 1.0, 1.0]

    !P.MULTI = 0

  ENDIF ELSE BEGIN

    !P.CHARSIZE = 1.0

    !P.REGION = !P.REGION - [0.0, ysiz, 0.0, ysiz]

    !P.MULTI = 0

    non_widget = 1

    win = !D.WINDOW

    no_erase = 1

  ENDELSE

  ; adjust charsize for plots

  ies_charsize

  ; check for overlap

  IF (NOT KEYWORD_SET(no_overlap)) AND (NOT KEYWORD_SET(non_widget)) THEN use_overlap = 1 ELSE use_overlap = 0

  ; if overlap used then adjust tstart and tstop

  IF use_overlap THEN BEGIN
; rcs_message,"overlap",/informational
    ; get previous conv

    conv = rcs_get_conv ( widget_id, /NO_COPY)

    ; adjust start time to pixel boundary

    start_pix = (output_info.tstart - overlap.tstart)/(overlap.tstop - overlap.tstart)*conv.xsize

    ; if not on pixel boundary then go to next pixel boundary
    ; NB if called again with the new output_info then tstart, tstop and start_pix
    ; will not be adjusted

    IF (start_pix MOD 1) GT 1.0D-4 THEN start_pix = LONG(start_pix)+1 ELSE start_pix = LONG(start_pix)

    ; adjust output_info start and stop to the next pixel boundary keeping time range the same

    output_info.tstart = overlap.tstart + start_pix/FLOAT(conv.xsize)*(overlap.tstop-overlap.tstart)

    output_info.tstop  = output_info.tstart + overlap.tstop-overlap.tstart

  ENDIF

  ; set up axes

  IF KEYWORD_SET(sectors) THEN BEGIN

    lo = 0

    IF output_info.showch THEN BEGIN
      hi = 360
      ytitle = 'Degrees'
    ENDIF ELSE BEGIN
      hi = 16
      ytitle = 'Sectors'
    ENDELSE

    ies_do_axes, output_info.tstart, output_info.tstop, lo, hi, xmargin, ymargin, $
                 XTITLE=output_info.xtitle, YTITLE=ytitle, STITLE=stitle, NO_ERASE=no_erase

  ENDIF ELSE IF KEYWORD_SET(hist) THEN BEGIN

    IF output_info.elog THEN lo = 1 ELSE lo = 0

    IF output_info.showch THEN BEGIN
      hi = 520
      ytitle = 'Energy [keV]'
    ENDIF ELSE BEGIN
      lo = lo/2.
      hi = 260
      ytitle = 'Energy [ADC bins]'
    ENDELSE

    ies_do_axes, output_info.tstart, output_info.tstop, lo, hi, xmargin, ymargin, $
                 XTITLE=output_info.xtitle, YTITLE=ytitle, LOG=output_info.elog, STITLE=stitle, NO_ERASE=no_erase

  ENDIF ELSE BEGIN

    lo = output_info.elow
    hi = output_info.ehigh

    ies_do_axes, output_info.tstart, output_info.tstop, lo, hi, xmargin, ymargin, $
                 XTITLE=output_info.xtitle, YTITLE=output_info.ytitle, LOG=output_info.elog, STITLE=stitle, $
                 NO_ERASE=no_erase

  ENDELSE

  ; check for overlap

  IF use_overlap THEN BEGIN
;message,"overlap",/informational
    ; get no of pixels overlap

    no_of_pixels = conv.xsize - start_pix

    ; transfer data

    IF no_of_pixels GT 0 THEN BEGIN

      temp = conv.freq[start_pix:*]
      ;temp = conv.freq[start_pix:*,*]

      conv.freq[0:no_of_pixels-1] = temp
      ;conv.freq[0:no_of_pixels-1,*] = temp

      temp = conv.data[start_pix:*,*]

      conv.data[0:no_of_pixels-1,*] = temp

    ENDIF

    IF no_of_pixels LT conv.xsize THEN BEGIN

      conv.freq[no_of_pixels:*] = 0.0
      ;conv.freq[no_of_pixels:*,*] = 0.0

      conv.data[no_of_pixels:*,*] = 0.0

    ENDIF

    ; transfer window image
    ; NB found that tvrd(0) did not read data correctly so use conv.image

    rcs_display_image, output_info, conv, 0, no_of_pixels-1, 0, conv.ysize-1

  ENDIF ELSE BEGIN

    ; check if rescaleable output device

    IF !D.FLAGS AND 1 THEN BEGIN

      ; fix array at some nominal size

      xorig = 0
      yorig = 0
      xsize = 512
      ysize = 128

      ; black out unused elements for postscript plot

      IF !D.NAME EQ 'PS' THEN TV, BYTARR(1,1), !X.WINDOW[0], !Y.WINDOW[0], $
        XSIZE=!X.WINDOW[1]-!X.WINDOW[0], YSIZE=!Y.WINDOW[1]-!Y.WINDOW[0], /NORMAL, ORDER=0

    ENDIF ELSE BEGIN

      ; get device coords of plot window

      vector = CONVERT_COORD ( !X.WINDOW, !Y.WINDOW, /NORMAL, /TO_DEVICE )

      ; get origin and size in device co-ords (inclusive of axes)

      xorig = vector(0,0)
      yorig = vector(1,0)
      xsize = FIX(vector(0,1) - xorig) ; removed + 1 because was getting overflow on widget plots 29/10/98
      ysize = FIX(vector(1,1) - yorig)

    ENDELSE

    ; create new conv
    ; for spectrogram data with overlap have 4 options :
    ;   1) overwrite data direct to display, retrieve overlap data from display
    ;   2) overwrite data in buffer, retrieve overlap data from buffer
    ;   3) sum data in buffer, keep column-by-column track of frequency
    ;   4) sum data in buffer, keep pixel-by-pixel track of frequency
    ; Options 3 and 4 are within acceptable range of speed, but memory requirements may become
    ; problematic. Option 3 results in some pixels having wrong frequency when energy binning
    ; (ie IES LUT) changes. In this circumstance it the data sum does not give much information
    ; anyway.
    ; Implement option 3 and comment out option 4 calculation.

    conv = rcs_conv_structure ( win, xorig, yorig, xsize, ysize)

    ; overplot axes again to neaten up
    ; NB dont use data coords because x axis extended in data coords
    ; NB dont want to use plot again because of multiple plots

    PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
           [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  ENDELSE

  ; save coord conversion

  rcs_set_conv, widget_id, conv, /NO_COPY

  ; get ctitle

  IF NOT KEYWORD_SET(ctitle) THEN ctitle = output_info.ztitle

  ; draw color bar

  ies_do_colorbar, output_info.minimum, output_info.maximum, CTITLE=ctitle, LOG=output_info.log

END
