;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PANEL_PEDESTAL_FILE
;
; Purpose     :	Outputs results of fits to text file.
;
; Explanation : Outputs results of fits to text file.
;
; Use         : <  >
;
; Inputs      :
;
; Opt. Inputs : None.
;
; Outputs     : None..
;
; Opt. Outputs:	None.
;
; Keywords    :
;
; Written     :	Version 0.0, 30/1/01
;
; Modified    : Version 0.1, 20/05/02
;                 Allowed no adjustment for temperature.
;                 Changed default name. Added time.
;               Version 0.2, 09/07/02
;                 Added printout of file name.
;               Version 0.3, 14/02/03
;                 Added creation of output directory if does not exist.
;               Version 0.4, 30/08/03
;                 Changed to RCS environment variables for TXT.
;               Version 0.5, 07/11/03
;                 Used rsc get subdir.
;
; Version     :	Version 0.5, 07/11/03
;-
;**********************************************************

PRO rcs_panel_pedestal_file, sc, modes, strips, w, temperatures, time, FILE=file

  xts = ['2 us', '5 us', '15 us', '50 us']

  ; set max temperature according to S/C

  CASE sc OF
    '1' : max_temp = 19.0
    '2' : max_temp = 20.4
    '3' : max_temp = 20.0
    '4' : max_temp = 21.2
    ELSE : max_temp = 20.0
  ENDCASE

  ; adjust w to max_temp

  aw = w

  IF KEYWORD_SET(temperatures) THEN BEGIN

    aw[0,*] = aw[0,*]-0.25*(temperatures-max_temp)

    aw[1,*] = aw[1,*]-0.1*(temperatures-max_temp)

    text = 'Pedestal info for adjusted for temperature '+rcs_strtrim(max_temp)+' degC'

  ENDIF ELSE BEGIN

    text = 'Pedestal info not adjusted for temperature'

  ENDELSE

  ; average over examples
  ; if there are 36 samples then interpret directly
  ; this avoids LUT assignment problems encountered

  ww = FLTARR(2,36)

  wi = REFORM(TRANSPOSE(INDGEN(4,9)),36)

  FOR k = 0, 35 DO BEGIN

    list = WHERE(strips EQ wi[k]/4 AND modes EQ wi[k] MOD 4, count)

    IF count EQ 1 THEN ww[*,k] = aw[*,list] $
    ELSE IF count GT 1 THEN ww[*,k] = TOTAL(aw[*,list],2)/count

  ENDFOR

  ; adjust pedestal position to stop contamination

  ; set pedestal position to its nominal value

  aww = NINT(ww)

  ; set minimum position of lower science channel
  ; allow margin of 1 bin

  thresholds = ww[0,*]+2*ww[1,*]+1

  ; adjust width to ensure that P+2S threshold is above thresholds

  diff = thresholds - aww[0,*] - 2*aww[1,*]

  aww[1,*] = aww[1,*] + FIX(2+diff)/2

  ; limit aww[1,*] to max 8 bins for B1=20

  aww[1,*] = aww[1,*] < 8

  ; adjust P to ensure that P+2S threshold is above thresholds
  ; if S unlimitted then this ony adjusts P by one bin at most

  diff = thresholds - aww[0,*] - 2*aww[1,*]

  aww[0,*] = aww[0,*] + FIX(diff-10)+10

  ; rearrange
  ; ww  contains mesaured Pedestal and Width
  ; aww contains P and S

  ww  = REFORM(ww,2,9,4)

  aww = REFORM(aww,2,9,4)

  ; output

  IF KEYWORD_SET(file) THEN BEGIN

    IF datatype(file) NE 'STRING' THEN BEGIN

      root = rcs_get_subdir('TXT',sc)

      IF NOT KEYWORD_SET(root) THEN RETURN

      ufile = concat_dir(root,'pedestals_'+anytim2cal ( time, FORM=8)+'.txt')

    ENDIF ELSE ufile = file

    ; print out

    rcs_text, 'Printing pedestal file : '+ufile, /OVERRIDE

    OPENW, unit, ufile, /GET_LUN

    PRINTF, unit, 'Time : '+anytim2cal ( time, FORM=8)

    PRINTF, unit, text

    PRINTF, unit, 'Pedestal positions [bins]'

    PRINTF, unit, ''

    PRINTF, unit, 'Detector', STRING(INDGEN(9)+1,FORMAT='(I6)'), FORMAT='(A12,9(",",A6))'

    FOR k=0,3 DO PRINTF, unit, xts[k], ww[0,*,k], FORMAT='(a12,9(",",F5.1))'

    PRINTF, unit, ''

    PRINTF, unit, 'Detector', STRING(INDGEN(9)+1,FORMAT='(I6)'), FORMAT='(A12,9(",",A6))'

    FOR k=0,3 DO PRINTF, unit, xts[k], aww[0,*,k], FORMAT='(a12,9(",",I5))'

    PRINTF, unit, ''

    PRINTF, unit, 'Detector', STRING(INDGEN(9)+1,FORMAT='(I6)'), FORMAT='(A12,9(",",A6))'

    FOR k=0,3 DO PRINTF, unit, xts[k], ww[1,*,k], FORMAT='(a12,9(",",F5.1))'

    PRINTF, unit, ''

    PRINTF, unit, 'Detector', STRING(INDGEN(9)+1,FORMAT='(I6)'), FORMAT='(A12,9(",",A6))'

    FOR k=0,3 DO PRINTF, unit, xts[k], aww[1,*,k], FORMAT='(a12,9(",",I5))'

    PRINTF, unit, ''

    PRINTF, unit, aww[0,0,*], FORMAT='("  p1 = [ [",I2,",",I2,",",I2,",",I2,"], $")'
    FOR k=1,7 DO PRINTF, unit, aww[0,k,*], FORMAT='("         [",I2,",",I2,",",I2,",",I2,"], $")'
    PRINTF, unit, aww[0,8,*], FORMAT='("         [",I2,",",I2,",",I2,",",I2,"] ]")'

    PRINTF, unit, ''

    PRINTF, unit, aww[1,0,*], FORMAT='("  s1 = [ [",I2,",",I2,",",I2,",",I2,"], $")'
    FOR k=1,7 DO PRINTF, unit, aww[1,k,*], FORMAT='("         [",I2,",",I2,",",I2,",",I2,"], $")'
    PRINTF, unit, aww[1,8,*], FORMAT='("         [",I2,",",I2,",",I2,",",I2,"] ]")'

    PRINTF, unit, ''

    FREE_LUN, unit

  ENDIF ELSE BEGIN

    PRINT, 'Time : '+anytim2cal ( time, FORM=8)

    PRINT, text

    PRINT, 'Pedestal positions [bins]'

    PRINT, ''

    PRINT, 'Detector', STRING(INDGEN(9)+1,FORMAT='(I6)'), FORMAT='(A12,9A6)'

    FOR k=0,3 DO PRINT, xts[k], ww[0,*,k], FORMAT='(a12,9F4.0)'

    PRINT, ''

    PRINT, 'Pedestal FWHM [bins]'

    PRINT, ''

    PRINT, 'Detector', STRING(INDGEN(9)+1,FORMAT='(I6)'), FORMAT='(A12,9A6)'

    FOR k=0,3 DO PRINT, xts[k], ww[1,*,k], FORMAT='(a12,9F4.0)'

    PRINT, ''

  ENDELSE

END
