;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PANEL_PLOT_CENTROID
;
; Purpose     :	Plots centroid of pedestal data in panel.
;
; Explanation :	Plots centroid of pedestal data in panel.
;               takes into account bin sizes.
;
; Use         : < rcs_panel_plot_centroid, widget_ids, header, data, output_info, detector_list, $
;                                    PSYM=psym, BM=bm >
;
; Inputs      : widget_ids   : LONG widget ids of draw window for each detector
;               header       : STRUCTURE IF data set header information
;               data         : STRUCTURE ARRAY IF data set data
;               output_info  : STRUCTURE containing display information
;               detector_list: LONARR list of detectors from IF data for plots 0:ndetecors-1
;
; Opt. Inputs : None.
;
; Outputs     : None..
;
; Opt. Outputs:	None.
;
; Keywords    : PSYM : INT plotting symbol to be used
;               BM   : FLAG indicates BM data
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 30/05/01
;                 Added common block for non-widget plots.
;               Version 0.2, 08/09/01
;                 Added SYMSIZE keyword.
;               Version 0.3, 25/10/03
;                 Used ies lut list.
;               Version 0.4, 21/11/03
;                 Changed argument list of rcs pedestals.
;                 Moved rcs get pedestal calibration.
;
; Version     :	Version 0.4, 21/11/03
;-
;**********************************************************

PRO rcs_panel_plot_centroid, widget_ids, header, data, output_info, detector_list, PSYM=psym, SYMSIZE=symsize, BM=bm

  ; get data in period of interest

  list = rcs_good_data ( header, data, output_info.tstart, output_info.tstop, NOBAD=output_info.nobad, COUNT=npoints)

  ; return if no data

  IF npoints EQ 0 THEN RETURN

  ; set psym

  IF NOT KEYWORD_SET(psym) THEN psym = 1

  ; get sector times

  x = rcs_x_sectors ( REFORM( [data[list].time, data[list].endtime], npoints, 2), header.nsectors)

  ; use only start time so independent of endtime

  x = [x[0,*]]

  ; remove offset in internal time

  x = x - output_info.tstart

  ; get data

  ndetectors = N_ELEMENTS(detector_list)

  array = REFORM(data[list].data[*,detector_list,*], header.nsectors, ndetectors, header.nbands, npoints)

  array = REFORM(TRANSPOSE(array, [2,1,0,3]), header.nbands, ndetectors, header.nsectors*npoints)

  IF KEYWORD_SET(bm) THEN BEGIN

    npeds = 4

    color = rcs_get_color(1)

  ENDIF ELSE BEGIN

    npeds = 2

    color = rcs_get_color(0)

  ENDELSE

  ; get pedestal info
  ; pedestals : FLTARR ( 5, ndetectors, npoints*nsectors)

  pedestals = rcs_pedestals ( rcs_extract_sc(HEADER=header), npeds, array, data[list].time, $
                              ies_lut_list(data[list].lut), header.nsectors, detector_list, BM=bm)

  ;print,TOTAL(pedestals[0,*,*],3)/npoints

  FOR d = 0, ndetectors-1 DO BEGIN

    ; retrieve co-ord conversion

    conv = rcs_get_conv(widget_ids[d])

    IF conv.win GE 0 THEN WSET, conv.win

    !X.S = conv.xcoords
    !Y.S = conv.ycoords
    !Y.TYPE = conv.log
    !X.WINDOW = conv.xwindow
    !Y.WINDOW = conv.ywindow
    !P.CLIP = conv.clip

    OPLOT, [x], [pedestals[1,d,*]], COLOR=rcs_get_color(0), PSYM=psym, SYMSIZE=symsize
    OPLOT, [x], [pedestals[2,d,*]], COLOR=rcs_get_color(1), PSYM=psym, SYMSIZE=symsize
    OPLOT, [x], [pedestals[3,d,*]], COLOR=rcs_get_color(2), PSYM=psym, SYMSIZE=symsize
    OPLOT, [x], [pedestals[4,d,*]], COLOR=rcs_get_color(3), PSYM=psym, SYMSIZE=symsize

    ; overplot to neaten
    ; NB noclip did not work well enough

    PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
             [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  ENDFOR

END
