;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PANEL_PLOT_HK
;
; Purpose     :	Plots HK data.
;
; Explanation :	Plots HK data.
;
; Use         : < rcs_panel_plot_hk, widget_ids, header, data, output_info, HK1=hk1, HK32=hk32>
;
; Inputs      : widget_ids   : LONG widget ids of draw window for each detector
;               header       : STRUCTURE IF data set header information
;               data         : STRUCTURE ARRAY IF data set data
;               output_info  : STRUCTURE containing display information
;
; Opt. Inputs : None.
;
; Outputs     : None..
;
; Opt. Outputs:	None.
;
; Keywords    : HK1    : FLAG indicates HK data
;               HK32   : FLAG indicates HK data
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 25/05/01
;                 Added NM/BM to HKFLAG plot.
;                 Changed to plotting at start time.
;                 Changed to using PSYM=3 for auto.
;               Version 0.2, 30/05/01
;                 Added common block for non-widget plots.
;
; Version     :	Version 0.2, 30/05/01
;-
;**********************************************************

PRO rcs_panel_plot_hk, widget_id, header, data, output_info, HK1=hk1, HK32=hk32

  ; get data in period of interest
  ; NB dont reject bad data -- this is what is being plotted

  list = rcs_good_data ( header, data, output_info.tstart, output_info.tstop, COUNT=npoints)

  ; return if no data

  IF npoints EQ 0 THEN RETURN

  ; remove offset in internal time
  ; and form 2 point array

  ; use only start time so independent of endtime

  x = [data[list].time]-output_info.tstart

  ; retrieve co-ord conversion

  conv = rcs_get_conv(widget_id)

  IF conv.win GE 0 THEN WSET, conv.win

  !X.S = conv.xcoords
  !Y.S = conv.ycoords
  !Y.TYPE = conv.log
  !X.WINDOW = conv.xwindow
  !Y.WINDOW = conv.ywindow
  !P.CLIP = conv.clip

  IF KEYWORD_SET(hk1) THEN BEGIN

    ; HK1 contains :
    ; 'ERISTAHV','ERISTOHV','ERIDEFHV','ERDTRIGM','ERDIFIND','ERDIFCAL','ERDDEADT','ERDHKFCR',
    ; 'ERISTACB','ERISTOCB','ERIENYCB','ERDTMMOD'

    ; extract erdtrigm

    y = data[list].data[0,3,0]

    ; plot erdtrigm

    OPLOT, [x], [y], PSYM=3, COLOR=rcs_get_color(2)

    ; extract erdifind

    y = data[list].data[0,4,0]

    ; plot erdifind

    OPLOT, [x], [y+6], PSYM=3, COLOR=rcs_get_color(3)

    ; generate y

    y = REPLICATE(1,npoints)

    ; extract erdifcal

    f = data[list].data[0,5,0]

    ; get set list

    setlist = WHERE ( f NE 0, count)

    ; plot unset bit

    OPLOT, [x], [8*y], PSYM=3, COLOR=rcs_get_color(0)

    ; plot where bit set

    IF count GT 0 THEN OPLOT, [x[setlist]], [8*y[setlist]], PSYM=3, COLOR=rcs_get_color(1)

    ; extract erddeadt

    f = data[list].data[0,6,0]

    ; get set list

    setlist = WHERE ( f NE 0, count)

    ; plot unset bit

    OPLOT, [x], [9*y], PSYM=3, COLOR=rcs_get_color(0)

    ; plot where bit set

    IF count GT 0 THEN OPLOT, [x[setlist]], [9*y[setlist]], PSYM=3, COLOR=rcs_get_color(1)

    ; check if new HK1 IF data

    IF (SIZE(data.data))[2] GE 12 THEN BEGIN

      ; extract erdtmmod

      f = data[list].data[0,11,0]

      ; get set list

      setlist = WHERE ( (f MOD 2) NE 0, count)

      ; plot unset bit

      OPLOT, [x], [11*y], PSYM=3, COLOR=rcs_get_color(0)

      ; plot where bit set

      IF count GT 0 THEN OPLOT, [x[setlist]], [11*y[setlist]], PSYM=3, COLOR=rcs_get_color(1)

    ENDIF

  ENDIF ELSE IF KEYWORD_SET(hk32) THEN BEGIN

    ; HK32 contains :
    ; 'ERDTOERC','ERDFRPRT','EREFXLUT'

    ; generate y

    y = REPLICATE(1,npoints)

    ; extract erefxlut
    ; NB auto = enabled for erefxlut=0

    f = 1-data[list].data[0,2,0]

    ; get set list

    setlist = WHERE ( f NE 0, count)

    ; plot unset bit

    OPLOT, [x], [10*y], PSYM=3, COLOR=rcs_get_color(0)

    ; plot where bit set

    IF count GT 0 THEN OPLOT, [x[setlist]], [10*y[setlist]], PSYM=3, COLOR=rcs_get_color(1)

  ENDIF

END

