;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PANEL_PLOT_LP
;
; Purpose     :	Plots data in lineplot format in panel.
;
; Explanation :	Plots data in lineplot format in panel.
;               Normally plot each detector/all energies on separate panel.
;               If SELECT then plot different detectors/1st energy on a single panel.
;
; Use         : < rcs_panel_plot_lp, widget_ids, header, data, output_info, detector_list, $
;                                    PSYM=psym, SELECT=select >
;
; Inputs      : widget_ids   : LONG widget ids of draw window for each detector
;               header       : STRUCTURE IF data set header information
;               data         : STRUCTURE ARRAY IF data set data
;               output_info  : STRUCTURE containing display information
;               detector_list: LONARR list of detectors from IF data for plots
;
; Opt. Inputs : None.
;
; Outputs     : None..
;
; Opt. Outputs:	None.
;
; Keywords    : SPIN_AVERAGE : FLAG indicates should sum over sectors before plotting
;                              else plots sector zero.
;               SHOW_SECTORS : FLAG indicates sectors expanded out along time axis
;               PSYM   : INT plotting symbol to be used
;               SELECT : FLAG indicates overplot detector list on same panel.
;               NOCAL  : FLAG indicating not to apply calibration factors
;               AVERAGE : produce hourly averages
;               ENERGY_LIST : INTARR list of energies.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 6/2/01
;                 Added AVERAGE keyword
;                 Added ENERGY_LIST keyword.
;               Version 0.2, 25/05/01
;                 Defaulted PSYM to 3.
;               Version 0.3, 30/05/01
;                 Added common block for non-widget plots.
;               Version 0.4, 08/09/01
;                 Added SYMSIZE keyword.
;               Version 0.5, 04/01/03
;                 Changed rcs_time_average to FUNCTION.
;               Version 0.6, 20/11/03
;                 Renamed routine rcs docal to rcs do cal.
;
; Version     :	Version 0.6, 20/11/03
;-
;**********************************************************

PRO rcs_panel_plot_lp, widget_ids, header, data, output_info, detector_list, ENERGY_LIST=energy_list, $
  SELECT=select, PSYM=psym, SYMSIZE=symsize, NOCAL=nocal, SPIN_AVERAGE=spin_average, SHOW_SECTORS=show_sectors, AVERAGE=average

  ; get data in period of interest

  list = rcs_good_data ( header, data, output_info.tstart, output_info.tstop, NOBAD=output_info.nobad, COUNT=npoints)

  ; return if no data

  IF npoints EQ 0 THEN RETURN

  ; get energy_list

  IF NOT KEYWORD_SET(energy_list) THEN energy_list = INDGEN(header.nbands)

  ; set psym

  IF NOT KEYWORD_SET(psym) THEN psym = 3

  ; use only start time so independent of endtime

  x = [data[list].time,data[list].endtime]

  ; y = [nsectors,ndetectors,nenergies,npoints]

  y = rcs_do_cal ( header, data[list], output_info, NOCAL=nocal)

  rcs_reform, x, y, l, header, npoints, RAT=output_info.rat, NOCAL=nocal, SPIN_AVERAGE=spin_average, SHOW_SECTORS=show_sectors

  x = x[0,*]

  y = y[detector_list,energy_list,*]

  IF KEYWORD_SET(average) THEN BEGIN

    IF NOT rcs_time_average( output_info.tstart, output_info.tstop, x, y, ax, ay) THEN RETURN

    ax = ax-output_info.tstart

  ENDIF ELSE BEGIN

    ; remove offset in internal time

    x = x-output_info.tstart

  ENDELSE

  ; plot channels using polyfill

  npanels = N_ELEMENTS(detector_list)

  ; check if plotting selected detectors in a single panel

  IF KEYWORD_SET(select) THEN npanels = 1

  FOR d = 0, npanels-1 DO BEGIN

    ; retrieve co-ord conversion

    conv = rcs_get_conv(widget_ids[d])

    IF conv.win GE 0 THEN WSET, conv.win

    !X.S = conv.xcoords
    !Y.S = conv.ycoords
    !Y.TYPE = conv.log
    !X.WINDOW = conv.xwindow
    !Y.WINDOW = conv.ywindow
    !P.CLIP = conv.clip

    ; loop over overlays

    IF KEYWORD_SET(average) THEN BEGIN

      IF KEYWORD_SET(select) THEN BEGIN

        FOR k = 0, N_ELEMENTS(detector_list)-1 DO OPLOT, [ax], [ay[k,0,*]], COLOR=rcs_get_color(k), PSYM=psym, SYMSIZE=symsize

      ENDIF ELSE BEGIN

        FOR k = 0, N_ELEMENTS(energy_list)-1 DO OPLOT, [ax], [ay[d,k,*]], COLOR=rcs_get_color(k), PSYM=psym, SYMSIZE=symsize

      ENDELSE

    ENDIF ELSE BEGIN

      IF KEYWORD_SET(select) THEN BEGIN

        FOR k = 0, N_ELEMENTS(detector_list)-1 DO OPLOT, [x], [y[k,0,*]], COLOR=rcs_get_color(k), PSYM=psym, SYMSIZE=symsize

      ENDIF ELSE BEGIN

        FOR k = 0, N_ELEMENTS(energy_list)-1 DO OPLOT, [x], [y[d,k,*]], COLOR=rcs_get_color(k), PSYM=psym, SYMSIZE=symsize

      ENDELSE

    ENDELSE

    ; overplot to neaten
    ; NB noclip did not work well enough

    PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
             [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  ENDFOR

END
