;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_PANEL_PLOT_SP
;
; Purpose     :	Plots data in spectral plots of counts per channel vs time.
;
; Explanation :	Plots data for a number of detectors in different draw windows.
;
; Use         : < rcs_panel_plot_sp, widget_ids, header, data, output_info, detector_list, $
;                                    SPIN_AVERAGE=spin_average, SHOW_SECTORS=show_sectors >
;
; Inputs      : widget_ids   : LONG widget ids of draw window for each detector
;               header       : STRUCTURE IF data set header information
;               data         : STRUCTURE ARRAY IF data set data
;               output_info  : STRUCTURE containing display information
;               detector_list: LONARR list of detectors from IF data for plots
;
; Opt. Inputs : None.
;
; Outputs     : None..
;
; Opt. Outputs:	None.
;
; Keywords    : SPIN_AVERAGE : FLAG indicates should sum over sectors before plotting
;                              else plots sector zero.
;               SHOW_SECTORS : FLAG indicates sectors expanded out along time axis
;               NOCAL        : FLAG indicating not to apply calibration factors
;               SECTORS      : FLAG indicates sector vs detector splot required.
;               ENERGY_RANGE : INTARR[2] upper and lower bounds of energy
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 30/05/01
;                 Added common block for non-widget plots.
;               Version 0.2, 24/10/03
;                 Used ies lut list.
;               Version 0.3, 20/11/03
;                 Used routine ies bins to kev..
;                 Renamed routine rcs docal to rcs do cal.
;
; Version     :	Version 0.3, 20/11/03
;-
;**********************************************************

PRO rcs_panel_plot_sp, widget_ids, header, data, output_info, detector_list, $
  SPIN_AVERAGE=spin_average, SHOW_SECTORS=show_sectors, NOCAL=nocal, SECTORS=sectors, ENERGY_RANGE=energy_range

  ; get data in period of interest

  list = rcs_good_data ( header, data, output_info.tstart, output_info.tstop, NOBAD=output_info.nobad, COUNT=npoints)

  ; return if no data

  IF npoints EQ 0 THEN RETURN

  ; remove offset in internal time

  x = [data[list].time,data[list].endtime]-output_info.tstart

  ; y = [nsectors,ndetectors,nenergies,npoints]

  y = rcs_do_cal ( header, data[list], output_info, NOCAL=nocal)

  l = data[list].lut

  rcs_reform, x, y, l, header, npoints, RAT=output_info.rat, NOCAL=nocal, SPIN_AVERAGE=spin_average, SHOW_SECTORS=show_sectors, SECTORS=sectors

  ; NB data resticted to minimum--> maximum in rcs_display_image

  IF KEYWORD_SET(sectors) THEN BEGIN

    ; extract required energies from data

    IF NOT KEYWORD_SET(energy_range) THEN energy_range = [0,header.nbands-1]

    ; make sure data undivided

    IF output_info.div AND (NOT KEYWORD_SET(nocal)) THEN BEGIN

      lut_list = ies_lut_list(l)

      ies_div, lut_list, header.nbands, header.ndetectors, ies_bins_to_kev(header.energy_cal, header.ch_positions), y, /REVERSE

      ; sum over energies

      y = TOTAL(y[*,*,energy_range[0]:energy_range[1],*],3)

      y = REFORM(y,header.nsectors,header.ndetectors,1,npoints)

      ch_positions = header.ch_positions[*,energy_range[0],*,*]

      ch_positions[1,0,*,*] =  header.ch_positions[1,energy_range[1],*,*]

      ies_div, lut_list, 1, header.ndetectors, ies_bins_to_kev(header.energy_cal, ch_positions), y

    ENDIF ELSE BEGIN

      ; sum over energies

      y = TOTAL(y[*,*,energy_range[0]:energy_range[1],*],3)

    ENDELSE

    y = REFORM(y, header.nsectors,header.ndetectors,npoints)

    ; define dummy positions

    output_chans = rcs_get_sectors(header.nsectors, SHOWCH=output_info.showch)

    ; set up dummy lut list

    lut_list = {st:0L, en:npoints-1L, lut:0}

    ; loop over detectors

    FOR d = 0, N_ELEMENTS(detector_list)-1 DO BEGIN

      ; retrieve co-ord conversion

      conv = rcs_get_conv(widget_ids[d], /NO_COPY)

      IF conv.win GE 0 THEN WSET, conv.win

      ; get data for detector

      rcs_mk_image, REFORM(y[*,detector_list[d],*],header.nsectors,npoints), x, lut_list, output_chans, conv, output_info

      ; store conv changes

      rcs_set_conv, widget_ids[d], conv, /NO_COPY

    ENDFOR

  ENDIF ELSE BEGIN

    ; set up output_chans

    output_chans = header.ch_positions

    ; set up energies in keV of channel boundaries
    ; NB include offset if present

    output_chans = output_chans*header.energy_cal[0]

    IF N_ELEMENTS(header.energy_cal) GT 1 THEN output_chans = output_chans + header.energy_cal[1]

    lut_list = ies_lut_list(l)

    ; loop over detectors

    FOR d = 0, N_ELEMENTS(detector_list)-1 DO BEGIN

      ; retrieve co-ord conversion

      conv = rcs_get_conv(widget_ids[d], /NO_COPY)

      IF conv.win GE 0 THEN WSET, conv.win

      ; get data for detector

      rcs_mk_image, REFORM(y[detector_list[d],*,*],header.nbands,npoints), x, lut_list, output_chans[*,*,detector_list[d],*], conv, output_info

      ; store conv changes

      rcs_set_conv, widget_ids[d], conv, /NO_COPY

    ENDFOR

  ENDELSE

END
