;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_PANEL_UPDATE
;
; Purpose     :	Updates panels.
;
; Explanation : Checks if data time still within panel range else redraws
;               panels. Updates panels.
;
; Use         : < timer = rcs_panel_update(panel_ids, output_info, positions) >
;
; Inputs      : panel_ids   : STRUCTURE containing widget ids of panels.
;               output_info : STRUCTURE containing output info for display
;               positions   : STRUCTURE containing positions of last data plotted
;
; Opt. Inputs : None.
;
; Outputs     : timer : FLOAT timer period.
;
; Opt. Outputs: None.
;
; Keywords    : REDRAW : Flag forcing redraw
;               WIN : Flag indicating windows output
;               GIF : Flag indicating GIF output
;               PNG : Flag indicating PNG output
;               PS  : Flag indicating postscript output
;               TRANGE :DBLARR(2) TAI start and stop times for plot.
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    :	Version 0.1, MKC, 10/12/00
;                 Added PNG format.
;                 Brought time range processing out of rcs_draw_spanel
;                 Fixed bug if rcs_start_plot returns 0 flag.
;               Version 0.2, 9/2/01
;                 Added NOT RESET to NO_OVERLAP
;               Version 0.3, 5/3/01
;                 Added tmin existence check.
;               Version 0.4, 30/05/01
;                 Added output_info and positions to argument list.
;               Version 0.5, 08/07/01
;                 Added keyword TRANGE.
;                 Added UPDATED.
;               Version 0.6, 20/07/01
;                 Added VERSION.
;               Version 0.7, 09/07/02
;                 Removed ncolors.
;
; Version     :	Version 0.7, 09/07/02
;-
;**********************************************************

FUNCTION rcs_panel_update, panel_ids, output_info, positions, REDRAW=redraw, $
  WIN=win, PS=ps, PNG=png, GIF=gif, TRANGE=trange

  COMMON rcs_extract_block, tmax, data_reset

  reset = KEYWORD_SET(data_reset) OR KEYWORD_SET(redraw)

  IF NOT KEYWORD_SET(tmax) THEN tmax = 0

  IF KEYWORD_SET(win) OR KEYWORD_SET(ps) OR KEYWORD_SET(png) OR KEYWORD_SET(gif) THEN nonwidget=1 ELSE nonwidget=0

  ; get data plot window
  ; NB .pre and .post not used below

  IF KEYWORD_SET(trange) THEN BEGIN

    tstart = trange[0]

    tstop  = trange[1]

  ENDIF ELSE IF NOT output_info.fulltime THEN BEGIN

    tstart = tmax + 60.0*output_info.pre

    tstop = tmax + 60.0*output_info.post

  ENDIF ELSE BEGIN

    ; display full time range excluding HK data

    tstop = rcs_get_tmax(NO_HK=output_info.nohk)

    tstart = rcs_get_tmin(NO_HK=output_info.nohk)

    IF tstart EQ 0 THEN tstart = tstop + output_info.pre*60.0 $
    ELSE tstart = tstart  < (tstop + output_info.pre*60.)

  ENDELSE

  ; set up non-widget display if required

  IF nonwidget THEN BEGIN

    ; get calendar day corresponding to midpoint

    heading = rcs_get_scname(rcs_extract_sc()) + ' ' + output_info.title + ' ' + anytim2cal ( (tstart+tstop)/2, FORM = 9, /DATE)

    info = rcs_start_plot( WIN=win, PS=ps, PNG=png, GIF=gif, HEADING=heading)

    IF NOT KEYWORD_SET(info) THEN GOTO, ending

  ENDIF

  ; if tmax is outside data plot window and data plot window has changed then redraw
  ; or if reset asserted

  IF ((tmax GE output_info.tstop) AND (tmax GT output_info.tmax)) OR reset THEN BEGIN

    ; set overlap
    ; overlap gives old time range for display
    ; output_info gives new time range for display

    overlap = {tstart:output_info.tstart, tstop:output_info.tstop}

    output_info.tstart = tstart
    output_info.tstop  = tstop
    output_info.tmax   = tmax

    ; check if new start time lies within old time range

    time_in_range = output_info.tstart GE overlap.tstart AND output_info.tstart LE overlap.tstop

    ; check if time 'range' unchanged -- this is not the same as the time period
    ; if time range changed then scaling is different and have to redraw

    time_range_unchanged = rcs_time_range_unchanged(overlap,output_info)

    ; check if overlap can be used

    IF time_range_unchanged AND time_in_range AND NOT reset THEN no_overlap = 0 ELSE no_overlap = 1

    ; set up axes
    ; NB output_info may be changed

    rcs_panel_draw, panel_ids, output_info, overlap, INFO=info, NO_OVERLAP=no_overlap

    ; reset plot positions for those panels not using overlap

    ; NB do not use (NOT time_range_unchanged) directly as KEYWORD

    IF reset OR NOT time_range_unchanged THEN data_reset = 1 ELSE data_reset=0

    rcs_get_positions, positions, /RESET, DATA_RESET=data_reset

  ENDIF

  ; update plots

  rcs_panel_update_plots, output_info, positions, panel_ids, UPDATED=updated, VERSION=version

  ; finish non-widget display

  IF nonwidget THEN rcs_end_plot, info, UPDATED=updated, VERSION=version

ending :

  ; set timer value

  timer = output_info.timer

  RETURN, timer

END
