;+
; Project     :	CLUSTER - RAPID
;
; Name        :	RCS_REFORM
;
; Purpose     :	Processes data into format for plotting.
;
; Explanation :	Processes data into format for plotting.
;
; Use         : < rcs_reform >
;
; Inputs      : times       : DBLARR[npoints,2] start and end times of each sample
;               data        : FLTARR[nsectors, ndetectors, nbands, npoints] data
;               l           : LONARR[npoints] l values
;               header      : STRUCURE containing data info
;               npoints     : LONG no of samples
;
; Opt. Inputs : None.
;
; Outputs     : times       : DBLARR[2,npoints] start and end times of each sample
;               data        : FLTARR[ndetectors, nbands, npoints] data
;               l           : LONARR[npoints] l values
;               npoints     : LONG new no of samples
;
; Opt. Outputs:	None.
;
; Keywords    : SPIN_AVERAGE : FLAG indicates should sum over sectors before plotting
;                              else plots sector zero.
;               SHOW_SECTORS : FLAG indicates sectors expanded out along time axis
;               SECTORS      : FLAG indicates sector plot
;               NOCAL        : FLAG indicating not to apply calibration factors
;               RAT          : FLAG indicating rate calibration
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    : Version 0.1, 13/12/00
;                 Fixed bug in leave which did not tranpose times by changing to sectors.
;                 Fixed bug in sectors which did not compensate for flux.
;
; Version     :	Version 0.1, 13/12/00
;-
;**********************************************************

PRO rcs_reform, times, data, l, header, npoints, RAT=rat, NOCAL=nocal, $
                SPIN_AVERAGE=spin_average, SHOW_SECTORS=show_sectors, SECTORS=sectors

  times = REFORM(times,npoints,2)

  data = REFORM(data, header.nsectors, header.ndetectors, header.nbands, npoints)

  ; NB must apply sectors in preference to spin_average

  IF KEYWORD_SET(sectors) THEN BEGIN

    ; keep sectors
    ; form 2 point array

    times = REFORM( TRANSPOSE(times), 2, npoints)

    IF (NOT KEYWORD_SET(nocal)) AND KEYWORD_SET(rat) AND (header.nsectors NE 1) THEN data = data * header.nsectors

  ENDIF ELSE IF KEYWORD_SET(spin_average) THEN BEGIN

    ; reform 2 point array
	; NB can spin_average counts and rates with total because rates
	; idiosyncratically stored as rates/nsectors

    times = REFORM( TRANSPOSE(times), 2, npoints)

    data = TOTAL(data,1)

  ENDIF ELSE IF KEYWORD_SET(show_sectors) THEN BEGIN

    data = TRANSPOSE(data,[1,2,0,3])

    data = REFORM(data, header.ndetectors, header.nbands, header.nsectors*npoints)

    ; form [2, npoints] array

    times = rcs_x_sectors( times, header.nsectors)

    IF KEYWORD_SET(l) THEN l = REBIN([l],npoints*header.nsectors,/SAMPLE)

    npoints = npoints*header.nsectors

    IF (NOT KEYWORD_SET(nocal)) AND KEYWORD_SET(rat) AND (header.nsectors NE 1) THEN data = data * header.nsectors

  ENDIF ELSE BEGIN

    ; use single sector

    ; form 2 point array

    times = REFORM(TRANSPOSE(times),2,npoints)

    data = REFORM(data[0,*,*,*], header.ndetectors, header.nbands, npoints)

    IF (NOT KEYWORD_SET(nocal)) AND KEYWORD_SET(rat) AND (header.nsectors NE 1) THEN data = data * header.nsectors

  ENDELSE

END
