;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_WIDGET_UPDATE
;
; Purpose     :	Updates current time display.
;
; Explanation : Updates current time display.
;
; Use         : < rcs_widget_update, widget_ids, output_info, positions >
;
; Inputs      : widget_ids : STRUCTURE ids of draw_widgets
;               output_info : STRUCTURE containing output info for display
;               positions   : STRUCTURE containing positions of last data plotted
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : REDRAW : Flag forcing redraw
;               WIN : Flag indicating windows output
;               GIF : Flag indicating GIF output
;               PNG : Flag indicating PNG output
;               PS  : Flag indicating postscript output
;               PLOTNO : FLAG indicating to query which plot to display
;
; Written     :	Version 0.0, 17/9/00
;
; Modified    :	Version 0.1, MKC, 13/12/00
;                 Added PNG format.
;                 Added margin to histogram plots.
;                 Added REDRAW keyword.
;                 Added PLOTNO KEYWORD
;                 Included temperature data.
;                 Added internal fix to save histogram data
;               Version 0.2, 30/05/01
;                 Added output_info and positions to argument list.
;               Version 0.3, 07/09/01
;                 Fixed bug with WIN keyword.
;               Version 0.4, 08/04/02
;                 Used rcs_check_ies.
;               Version 0.5, 20/05/02
;                 Corrected bug in batch processing plots.
;                 Enabled pedestal file production.
;                 Added HTML output.
;                 Improved mechanism for starting new page.
;               Version 0.6, 09/07/02
;                 Only output pedestal txt file if producing nonwidget output.
;                 Removed ncolors.
;                 Redrew border and changed plot symbol size.
;
; Version     :	Version 0.6, 09/07/02
;-
;**********************************************************

FUNCTION rcs_widget_update, widget_ids, output_info, positions, $
  WIN=windows, PS=ps, PNG=png, GIF=gif, REDRAW=redraw, PLOTNO=plotno

  COMMON rcs_ehist_block, ehist_header, ehist_data

  COMMON rcs_hk8_block, hk8_header, hk8_data

  COMMON rcs_extract_block, tmax, data_reset

  ;COMMON rcs_temporary, pedestal_data

  IF KEYWORD_SET(windows) OR KEYWORD_SET(ps) OR KEYWORD_SET(png) OR KEYWORD_SET(gif) THEN nonwidget=1 ELSE nonwidget=0

  IF KEYWORD_SET(data_reset) OR KEYWORD_SET(redraw) THEN BEGIN

    IF NOT KEYWORD_SET(nonwidget) THEN BEGIN

      ; if common blocks have been reset then redraw

      WIDGET_CONTROL, widget_ids.hist, GET_VALUE=win

      WSET, win

      ERASE

    ENDIF

    ; reset plot positions for all panels
    ; NB ehist uses overlap in spectral plot so need /DATA_RESET

    rcs_get_positions, positions, /RESET, /DATA_RESET

  ENDIF

  ; update plots

  IF output_info.hist NE 0 AND KEYWORD_SET(ehist_header) THEN BEGIN

    x = FINDGEN(256)+0.5

    list = rcs_get_list( ehist_header, ehist_data, positions.ehist, new_position, count)

    ; get good data

    IF count GT 0 THEN list = rcs_good_data ( ehist_header, ehist_data, dummy_tstart, dummy_tstop, NOBAD=output_info.nobad, LIST=list, COUNT=count)

    ; fudge to account for wrongly assigned mode
    ; NB if bad mode then not plotted

    ncount = 0
    nlist = list

    FOR k = 0, count-1 DO BEGIN

      ; get mode = 0-3

      dummy = rcs_check_ies(mode, ehist_data[list[k]].data, /HIST, /CHECK)

      ; check mode

      IF mode NE ehist_data[list[k]].lut MOD 4 THEN BEGIN

        rcs_text, 'Integration mode error : '+STRING(ehist_data[list[k]].lut MOD 4)+' '+STRING(mode)

      ENDIF ELSE BEGIN

        ; add to list

        nlist[ncount] = list[k]

        ncount = ncount + 1

      ENDELSE

    ENDFOR

    count = ncount

    IF count GT 0 THEN list = nlist[0:count-1]

    ; check if plot select required

    IF count GT 0 AND KEYWORD_SET(plotno) THEN list = rcs_select_plot(widget_ids.base, ehist_header, ehist_data, LIST=list, COUNT=count)

    ; check if anything to plot

    IF count GT 0 THEN BEGIN

      ; get S/C

      sc = rcs_extract_sc()

      ; set up histogram bin positions

      x = 0.5 + FINDGEN(256)

      ; get data to be plotted

      temp_data = ehist_data[list]

      ; sum over similar integration times

      IF 0 THEN BEGIN

        list = [-1]

        FOR k=0,35 DO BEGIN

          tlist = WHERE(temp_data.lut EQ k, tcount)

          IF tcount GT 1 THEN temp_data[tlist[0]].data = TOTAL(temp_data[tlist].data,4)

          IF tcount GT 0 THEN list = [list,tlist[0]]

        ENDFOR

        count = N_ELEMENTS(list)-1

        IF count GT 0 THEN BEGIN

          list = list[1:*]

          list = list[SORT(list)]

          temp_data = temp_data[list]

        ENDIF

      ENDIF

      ; get pedestal fits

      w = rcs_fit(REFORM(temp_data.data,256,count), XMAX=100, /QUIET)

      ; get temperatures

      IF KEYWORD_SET(hk8_header) THEN temperatures = INTERPOL( REFORM(hk8_data.data[0,6,0]), hk8_data.time, temp_data.time)

      ; output pedestal file
      ; set up hist html file name

      IF nonwidget THEN BEGIN

        rcs_panel_pedestal_file, sc, temp_data.lut MOD 4, temp_data.lut/4, w, temperatures, ehist_header.tstart, /FILE

        ; set hist html file

        html_file = concat_dir(rcs_get_subdir( 'TXT', sc),'histogram_html.txt')

        rcs_text, 'Updating hist html file :'+html_file, /OVERRIDE

      ENDIF

      ; IF KEYWORD_SET(pedestal_data) THEN pedestal_data = [[pedestal_data],[FIX(sc),mode,strip,temperature,w[*,k]]] $
      ;  ELSE pedestal_data = [FIX(sc),mode,strip,temperature,[*,k]]

      current_mode = -1

      previous_strip = 9

      FOR k = 0, count-1 DO BEGIN

        ehist = temp_data[k]

        strip = ehist.lut/4

        mode = temp_data[k].lut MOD 4

        ; if bad mode then miss out

        IF mode NE -1 THEN BEGIN

          ; set up non-widget display if required

          IF nonwidget THEN BEGIN

            IF NOT KEYWORD_SET(info) OR strip LE previous_strip OR mode NE current_mode THEN BEGIN

              ; finish previous

              IF KEYWORD_SET(info) THEN rcs_end_plot, info, /UPDATED

              current_mode = mode

              ; get temperature

              IF KEYWORD_SET(hk8_header) THEN $
                xtemp = ' @'+rcs_strtrim(temperatures[k],FORMAT='(F5.1)')+'C' $
              ELSE xtemp = ''

              ; get filename

              IF NOT rcs_hist_filename( sc, ehist.time, mode, PS=ps, PNG=png, GIF=gif ) THEN RETURN, 0

              ; output html to log file

              rcs_hist_html, sc, html_file, ehist.time, mode, PS=ps, PNG=png, GIF=gif, WIN=windows

              heading = rcs_get_scname(sc) + ' ' + output_info.title + ' ' + $
                 (['2 us','5 us', '15 us', '50 us'])[mode] + ' ' + anytim2cal ( ehist.time) + xtemp

              info = rcs_start_plot( WIN=windows, PS=ps, PNG=png, GIF=gif, HEADING=heading)

              ncolumns = 3

              xoff = 0.02

              xsiz = (1.0-2*xoff)/ncolumns

              ysiz = info.ysiz/3.0

              ytop = info.ytop

            ENDIF

            ; set plot region

            !P.REGION = [ xoff+(strip MOD ncolumns)*xsiz, ytop-ysiz*(strip/ncolumns+1), $
                          xoff+(strip MOD ncolumns)*xsiz+xsiz, ytop-ysiz*(strip/ncolumns)]

            ; do plot

            PLOT, x, ehist.data, $
              YRANGE=[output_info.minimum, output_info.maximum], $
              XRANGE=[output_info.elow, output_info.ehigh], $
              YLOG=output_info.log, XLOG=output_info.elog, $
              TITLE='IES'+rcs_strtrim(strip)+' '+anytim2cal(ehist.time), $
              XMARGIN=[5,0], /NOERASE, /NODATA, CHARSIZE=0.7*!P.CHARSIZE

          ENDIF ELSE BEGIN

            ; widget based output

            WIDGET_CONTROL, widget_ids.hist, GET_VALUE=win

            WSET, win

            !P.REGION = 0

            !P.POSITION = 0

            IF strip LE previous_strip OR mode NE current_mode THEN BEGIN

              current_mode = mode

              ERASE

            ENDIF

            ; NB the first element of !P.MULTI gives the no. of plot sectors remaining
            ; If 0 then erases plot before starting
            ; i.e 0,9 puts plot at top left
            ;     8 puts plot at 2nd from top left
            ;     1 puts plot at bottom right

            !P.MULTI = [9-strip, 3, 3]

            PLOT, x, ehist.data, $
              YRANGE=[output_info.minimum, output_info.maximum], $
              XRANGE=[output_info.elow, output_info.ehigh], $
              YLOG=output_info.log, XLOG=output_info.elog, $
              TITLE='IES'+rcs_strtrim(strip)+' '+anytim2cal(ehist.time), CHARSIZE=1.5, /NODATA

          ENDELSE

          IF NOT output_info.showch THEN BEGIN

            ; draw bin positions using dotted line
            ; dont want to draw lines more than once else looks bad
            ; draw sample averages over bins with solid line

            ; get samples
            ; NB mode is 0-3

            samples = rcs_resample_hist(sc, ehist.time, strip, mode, ehist.data, BIN_POSITIONS=bin_positions, NO_OF_ENERGIES=no_of_energies)

            ; divide by bin size

	  	    samples = samples/REFORM(bin_positions[1,*]-bin_positions[0,*])

            ; plot samples

            FOR s = 0, no_of_energies-1 DO $
              POLYFILL, bin_positions[[0,0,1,1],s],([output_info.minimum,samples[s]])[[0,1,1,0]], COLOR=150, NOCLIP=0

            ;OPLOT, PSYM=-2, x, ehist.data

            bins = REFORM(bin_positions,2*no_of_energies)

            ubins = bins(UNIQ(bins))

            FOR b = 0, N_ELEMENTS(ubins)-1 DO $
              OPLOT, [ubins[b],ubins[b]], [output_info.minimum,output_info.maximum], LINESTYLE=1

            ; draw in science channel threshold

            OPLOT, [bin_positions[0,6],bin_positions[0,6]], [output_info.minimum,output_info.maximum], LINESTYLE=0

            ; overplot axes again to neaten up

            PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
                   [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

          ENDIF

          ; plot data

          OPLOT, x, ehist.data, PSYM=-2, SYMSIZE=0.4

          ; update previous_strip only if good data

          previous_strip = strip

        ENDIF ELSE BEGIN

          ; as stands should not get any bad modes

          rcs_text, 'Omitted histogram plot'

        ENDELSE

      ENDFOR

      ; finish previous

      IF KEYWORD_SET(info) THEN rcs_end_plot, info, /UPDATED

      positions.ehist = new_position

    ENDIF

  ENDIF

  ; return timer

  RETURN, output_info.timer

END
