;+
; Project     :	CLUSTER II/RAPID
;
; Name        :	rcs_4sc_plot
;
; Purpose     :	Produces 4 spacecraft survey plot panel.
;
; Explanation : Produces 4 spacecraft survey plot panel.
;
; Use         : < rcs_4sc_plot, day, WINDOWS=windows, PS=postscript, PNG=png, GIF=gif, BWPS=bwps, $
;                             TYPE=type >
;
; Inputs      : day  : STRING day in form yymmdd.
;
; Opt. Inputs :
;
; Outputs     : None.
;
; Opt. Outputs:	None.
;
; Keywords    : /WINDOWS    = plots output in x window.
;               /PS         = plots output as postscript file
;                                if string then assumes it is a file name
;               /PNG        = plots output as PNG file
;                                if string then assumes it is a file name
;               /GIF        = plots output as GIF file
;                                if string then assumes it is a file name
;               /BWPS       = plots output as black and white postscript file
;                                if string then assumes it is a file name
;               TYPE        = INT giving type of survey plot required.
;                                0 : IES/IPS standard plot
;                                1 : HISTE/IES/IPS pitch angle plot
;               WRAP        = FLOAT indicating should wrap data in plot window and wrap value
;
; Written     :	Version 0.0, Martin Carter, RAL, 16/2/01
;
; Modified    :	Version 0.1, 14/03/01
;                 Modified plots and output produced.
;                 Removed ids from rcs_text.
;               Version 0.2, 16/03/01
;                 Changed to use WYSIWYG.
;               Version 0.3, 25/04/01
;                 Allowed multiple elements in title.
;                 Added ytick names.
;               Version 0.4, 26/03/02
;                 Added YPOS and YSIZ keywords.
;               Version 0.5, 10/07/02
;                 Removed ncolors, COLTAB.
;               Version 0.6, 14/1/03
;                 Used routine rcs_plot_end.
;               Version 0.7, 28/03/03
;                 Used ies_end_plot.
;               Version 0.8, 24/09/03
;                 Removed draw widget block,
;               Version 0.9, 19/01/04
;                 Added SHIFT and WRAP keywords and protected against floating point underflows.
;
; Version     :	Version 0.9, 19/01/04
;-
;**********************************************************

PRO rcs_4sc_plot, devname, file, tstart, tend, atime, adata, adata1, adata2, adata3, adata4, $
                          WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, GIF=gif, BWPS=bwps, $
                          PG_POS=pg_pos, NO_XAXIS=no_xaxis, NO_END=no_end, SHRINK=shrink, SETUP=setup, NPANELS=npanels, $
                          HEADING=heading, FIXPANSIZ=fixpansiz,$
                          XMARGIN=xmargin, MINIMUM=minimum, MAXIMUM=maximum, TITLE=title, $
                          YTICKV=ytickv, ytickname=ytickname, SHIFT=shift, YPOS=ypos, YSIZ=ysiz, WRAP=wrap

  ; plot already set up in ies_window_plot

  IF NOT KEYWORD_SET(pg_pos) THEN rcs_message, /IMPLEMENTATION

  ; set default plot type

  IF NOT KEYWORD_SET(postscript) AND NOT KEYWORD_SET(png) AND NOT KEYWORD_SET(gif) AND NOT KEYWORD_SET(bwps) THEN windows=1

  ; set no_xaxis

  IF NOT KEYWORD_SET(no_xaxis) THEN no_xaxis=0

  ; setup npanels

  IF NOT KEYWORD_SET(npanels) THEN npanels = 6

  ; set min and max

  IF NOT KEYWORD_SET(minimum) THEN minimum = 1e3

  IF NOT KEYWORD_SET(maximum) THEN maximum = 1e6

  ; check if margins defined

  IF N_ELEMENTS(xmargin) EQ 0 THEN BEGIN

    ; get standard character sizes in normal coords

    x_chsize = get_chsize(/X)
    y_chsize = get_chsize(/Y)

    ; fix margins so that in postcript format plot size is 16 cm

    xmargin = [10.0,16.0]

    xmargin = 3.9/19.9/x_chsize * xmargin/TOTAL(xmargin)

  ENDIF

  IF N_ELEMENTS(ymargin) EQ 0 THEN BEGIN

    ymargin = [0,0.5]

  ENDIF

  IF KEYWORD_SET(ytickv) THEN BEGIN

    no_yaxis = 1

    IF NOT KEYWORD_SET(ytickname) THEN ytickname = rcs_strtrim(ytickv)

  ENDIF

  ; get character sizes in normal coords

  x_chsize = get_chsize(/X)
  y_chsize = get_chsize(/Y)

  ; specify ytop

  ytop = 1-2.5*y_chsize

  IF NOT KEYWORD_SET(fixpansiz) THEN BEGIN

    ysiz = (ytop-y_chsize*0.7)/npanels

  ENDIF ELSE BEGIN

    ysiz = (ytop-y_chsize*0.7-1.5*y_chsize)/npanels

  ENDELSE

  ypos = ytop - ysiz*pg_pos

  ; do page display

  !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]  - pg_pos*[ 0.0, ysiz, 0.0, ysiz]

  ; set up time_start and time_end

  time_start = tstart
  time_end   = tend

  ; set up axes

  ies_do_axes, time_start, time_end, minimum, maximum, xmargin, ymargin, NO_YAXIS=no_yaxis, $
               XTITLE=xtitle, YTITLE='[Cts/s/cm2/sr]', NO_XAXIS=no_xaxis, /NO_ERASE, /LOG, SHRINK=shrink

  ; plot y axis

  IF KEYWORD_SET(no_yaxis) THEN BEGIN

    AXIS, YAXIS=0, YMINOR=9, /YLOG, YTICKLEN=-x_chsize, YTICKNAME=ytickname, YTICKS=N_ELEMENTS(ytickv)-1, YTICKV=ytickv, CHARSIZE=!P.CHARSIZE*0.8

  ENDIF

  ; do SC

  times = atime - time_start

  ; NB protect against data for SC not set up
  ; remember these are arrays

  IF KEYWORD_SET(shift) THEN BEGIN

    IF KEYWORD_SET(adata1) THEN rcs_sc_plot, times, 10*adata1/(adata > 1e-10), COLOR=rcs_get_color(0)

    IF KEYWORD_SET(adata2) THEN rcs_sc_plot, times, adata2/(adata > 1e-10), COLOR=rcs_get_color(1)

    IF KEYWORD_SET(adata3) THEN rcs_sc_plot, times, 0.1*adata3/(adata > 1e-10), COLOR=rcs_get_color(2)

    IF KEYWORD_SET(adata4) THEN rcs_sc_plot, times, 0.01*adata4/(adata > 1e-10), COLOR=rcs_get_color(3)

  ENDIF ELSE BEGIN

    IF KEYWORD_SET(adata1) THEN rcs_sc_plot, times, adata1, COLOR=rcs_get_color(0)

    IF KEYWORD_SET(adata2) THEN rcs_sc_plot, times, adata2, COLOR=rcs_get_color(1)

    IF KEYWORD_SET(adata3) THEN rcs_sc_plot, times, adata3, COLOR=rcs_get_color(2)

    IF KEYWORD_SET(adata4) THEN rcs_sc_plot, times, adata4, COLOR=rcs_get_color(3)

  ENDELSE

  ; wrap data

  IF KEYWORD_SET(wrap) THEN BEGIN

    IF KEYWORD_SET(adata1) THEN rcs_sc_plot, times, adata1/wrap, COLOR=rcs_get_color(0)

    IF KEYWORD_SET(adata2) THEN rcs_sc_plot, times, adata2/wrap, COLOR=rcs_get_color(1)

    IF KEYWORD_SET(adata3) THEN rcs_sc_plot, times, adata3/wrap, COLOR=rcs_get_color(2)

    IF KEYWORD_SET(adata4) THEN rcs_sc_plot, times, adata4/wrap, COLOR=rcs_get_color(3)

  ENDIF

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  ; set up y position of labels

  yp = !P.POSITION(3) - 1.25*y_chsize

  IF KEYWORD_SET(title) THEN BEGIN

    ; write out title
    ; NB use !p.region here rather than position to give more room

    FOR  k= 0, N_ELEMENTS(title)-1 DO BEGIN

      XYOUTS, (!P.POSITION(2)+!P.REGION(2) - STRLEN(title[k])*x_chsize)/2.0, yp, title[k], /NORMAL, CHARSIZE=!P.CHARSIZE

      yp = yp - 1.5*y_chsize

    ENDFOR

  ENDIF


  labels = ['SC1','SC2','SC3','SC4']

  ; draw band labels

  ; a space has been left to the left of the plot window
  ; reset plot window to point at this space
  ; adjust plot window by no. of characters

  !P.POSITION(0) = !P.POSITION(2) + 1*x_chsize
  !P.POSITION(2) = !P.REGION(2) - 1*x_chsize

  ; set output positions for band labels centred in plot window in normal coords

  ; get max band label length, add 3 for line

  max_length = 3 + MAX ( STRLEN ( labels ) )

  ; center labels

  xp = (!P.POSITION(0) + !P.POSITION(2))/2.0 - max_length*x_chsize/2.0

  yp = yp-y_chsize/4.0

  ; loop over labels

  FOR k = 0, N_ELEMENTS(labels)-1 DO BEGIN

    ; plot labels

    PLOTS, [xp, xp+2*x_chsize], [yp, yp], /NORMAL, COLOR=rcs_get_color(k)
    XYOUTS, xp+3.0*x_chsize, yp, labels(k), COLOR=rcs_get_color(k), /NORMAL, CHARSIZE=!P.CHARSIZE

    ; adjust yp

    yp = yp - 1.5*y_chsize

  ENDFOR

  ; finish off

  IF NOT KEYWORD_SET(no_end) THEN ies_end_plot, file, devname, WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, GIF=gif, BWPS=bwps  $
  ELSE BEGIN

      ; set !p.region and position back again

      !P.REGION   = 0
      !P.POSITION = 0

  ENDELSE

END
