;+
; Project     :	CLUSTER II/RAPID
;
; Name        :	rcs_sc_average
;
; Purpose     :	Produces data average. Calibrates data. Integrates over energies.
;
; Explanation : Produces data average. Calibrates data. Integrates over energies.
;               Input data is spin-averaged, detector-averaged counts.
;               Returns 1 if OK else 0.
;
; Use         : < res = rcs_sc_average( time_start, time_end, header, data, atime, adata, $
;                   PERIOD=period, INCLUDE=include, NOBAD=nobad, MEDIAN=median, POINTS=points, THRESHOLD=threshold) >
;
; Inputs      : time_start : DOUBLE TAI start time
;               time_end   : DOUBLE TAI stop time
;               header     : STRUCTURE containing data info.
;               data       : STRUCTURE ARRAY containing data counts.
;
; Opt. Inputs : None.
;
; Outputs     : Returns 1 if OK else 0.
;
; Opt. Outputs: atime : DBLARR[] TAI time of each resampled data point (start time).
;               data  : FLOAT[] calibrated, averaged, accumulated data.
;
; Opt. Outputs:	None.
;
; Keywords    : PERIOD  : LONG time span in seconds. Only truncates to one day.
;               INCLUDE : FLAG indicating whether bottom energy channel included in data set
;                         If only one energy band then disregarded.
;               NOBAD   : FLAG indicating whether data flagged as bad included in data set
;               MEDIAN  : FLAG indicating MEDIAN filter applied to datra within each time sample.
;               POINTS  : LONARR[] returning no. of points in each new sample.
;               MEDIAN  : FLAG indicating should apply median filter rather than average.
;               THRESHOLD : FLAG indicating should check if any data above threshold before averaging
;
; Written     :	Version 0.0, Martin Carter, RAL, 16/2/01
;
; Modified    :	Version 0.1, 24/04/01
;                 Included integration over energies withint this routine.
;               Version 0.2, 07/02/02
;                 Added NOBAD keyword.
;                 Previously applied as default.
;               Version 0.3, 13/02/02
;                 Added INCLUDE keyword.
;               Version 0.4, 21/03/02
;                 Added MEDIAN and POINTS keyword.
;                 Allowed multiple detectors.
;               Version 0.5, 09/04/02
;                 Coped with case where only single point.
;               Version 0.6, 03/05/02
;                 Explicitly set div = 0.
;               Version 0.7, 03/05/02
;                 Added THRESHOLD keyword.
;               Version 0.8, 04/01/03
;                 Updated documentation.
;                 Changed rcs_time_average to FUNCTION.
;                 Changed so that deals with nbands=1 and NOT INCLUDE.
;               Version 0.9, 20/11/03
;                 Renamed routine rcs docal to rcs do cal.
;
; Version     :	Version 0.9, 20/11/03
;-
;**********************************************************

FUNCTION rcs_sc_average, time_start, time_end, header, data, atime, adata, $
  PERIOD=period, NOBAD=nobad, INCLUDE=include, MEDIAN=median, POINTS=points, THRESHOLD=threshold

  IF NOT KEYWORD_SET(header) THEN RETURN, 0

  ; get data in period of interest

  list = rcs_good_data ( header, data, time_start, time_end, NOBAD=nobad, COUNT=npoints)

  ; return if no data

  IF npoints EQ 0 THEN RETURN, 0

  ; set up output_info

  output_info = rcs_output_info()

  output_info.rat = 1
  output_info.cal = 1
  output_info.eff = 1
  output_info.div = 0

  x = [data[list].time,data[list].endtime]

  ; calibrate data

  y = rcs_do_cal ( header, data[list], output_info) ; y = [nsectors,ndetectors,nenergies,npoints]

  ; reform

  rcs_reform, x, y, l, header, npoints, RAT=output_info.rat  ; y = [ndetectors,nenergies,npoints]

  ; sum counts/cm2/sr/sec over energies
  ; NB brackets needed if only single point ?!

  IF header.nbands EQ 1 THEN $
    y = REFORM([y],header.ndetectors,1,npoints) $
  ELSE IF KEYWORD_SET(include) THEN $
    y = REFORM([TOTAL(y,2)],header.ndetectors,1,npoints) $
  ELSE $
    y = REFORM([TOTAL(y[*,1:*,*],2)],header.ndetectors,1,npoints)

  x = x[0,*]

  ; time average

  RETURN, rcs_time_average( time_start, time_end, x, y, atime, adata, PERIOD=period,$
                      MEDIAN=median, POINTS=points, THRESHOLD=threshold)

END
