;+
; Project     :	POLAR - CEPPAD
;
; Name        :	rcs_survey_extract
;
; Purpose     :	Extracts required data.
;
; Explanation : Extract data.
;
; Use         : < res = rcs_survey_extract(time_range, detectors, FORMAT=format, MODES=modes) >
;
; Inputs      : time_range : DOUBLE[2] TAI time range.
;               detectors  : INTARR(input_header.ndetectors) flags indicating whether to use detectors or not.
;
; Opt. Inputs : None.
;
; Outputs     : Updates header and data structures in COMMON block.
;               res : INT 0 : error, 1 : OK,
;
; Opt. Outputs:	None.
;
; Keywords    : FORMAT : string indicating plot required.
;               MODES  : STRARR specifying data modes allowed ie 'ABC'
;               LB     : INTARR lower bounds of energy channels
;               UB     : INTARR upper bounds of energy channels
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/11/97
;
; Modified    :    Version 0.1, MKC, 18/2/98
;                    Added HISTe, HISTp processing.
;                    Added RANGE keyword.
;                  Version 0.2, 7/10/98
;                    Added Magnetic field and attitude arrays to common block and to ies extdata
;                    Allowed detector sum.
;                  Version 0.3, 30/10/98
;                    Separated calibrate stuff.
;                    Remove bottom IES channel.
;                  Version 0.4, 28/2/01
;                    Updated for RAPID.
;                  Version 0.5, 03/05/01
;                    Added E0 keyword.
;                  Version 0.6, 07/07/01
;                    Dealt with insufficient points from ies extdata.
;                    Changed to function.
;                    Silenced error messages.
;                  Version 0.7, 14/05/02
;                    Added keywords LB and UB and removed E0.
;
; Version     :	Version 0.7, 14/05/02
;-
;**********************************************************

FUNCTION rcs_survey_extract, time_range, detectors, FORMAT=format, MODES=modes, LB=lb, UB=ub

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information  ** used
  ; data          = structure array containing extracted data               ** used
  ; output_info   = structure containing plotting and processing info       ** used
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, input_header, input_data, header, data, $
           output_info, output_chans, output_array, ids, dnfull, dnscroll, files, ephem, mf, at

  IF NOT KEYWORD_SET(format) THEN format = 'ENERGY vs DETECTOR'

  IF format EQ 'ENERGY vs DETECTOR' THEN BEGIN

    nbands = 0

    IF NOT KEYWORD_SET(lb) THEN lb = INDGEN(input_header.nbands) ELSE nbands = N_ELEMENTS(lb)

    IF NOT KEYWORD_SET(ub) THEN ub = INDGEN(input_header.nbands) ELSE nbands = N_ELEMENTS(ub)

    ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:REPLICATE(1,N_ELEMENTS(lb)), nbands:nbands, botchan:0, topchan:input_header.nbands-1, $
                   lb:lb, ub:ub }, $
                   input_header, input_data, 0, header, data, errmsg, EPHEM=ephem, MF=mf, AT=at, MODES=modes

    IF errmsg NE '' THEN BEGIN
      IF NOT rcs_get_silent() THEN rcs_message, errmsg, /INFORMATIONAL
      RETURN,0
    ENDIF

  ENDIF ELSE IF format EQ 'DETECTOR vs SECTOR' THEN BEGIN

    IF NOT KEYWORD_SET(lb) THEN lb = [0]

    IF NOT KEYWORD_SET(ub) THEN ub = [input_header.nbands-1]

    ; want to create single energy band summed over all energies
    ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:[1], nbands:1, botchan:0, topchan:0, $
                   lb:lb, ub:ub }, $
                   input_header, input_data, 0, header, data, errmsg, EPHEM=ephem, MF=mf, AT=at, MODES=modes

    IF errmsg NE '' THEN BEGIN
      IF NOT rcs_get_silent() THEN rcs_message, errmsg, /INFORMATIONAL
      RETURN,0
    ENDIF

  ENDIF ELSE IF format EQ 'SECTOR vs DETECTOR' THEN BEGIN

    IF NOT KEYWORD_SET(lb) THEN lb = [0]

    IF NOT KEYWORD_SET(ub) THEN ub = [input_header.nbands-1]

    ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:[1], nbands:1, botchan:0, topchan:0, lb:lb, ub:ub }, $
                   input_header, input_data, 0, header, data, errmsg, EPHEM=ephem, MF=mf, AT=at, MODES=modes

    IF errmsg NE '' THEN BEGIN
      IF NOT rcs_get_silent() THEN rcs_message, errmsg, /INFORMATIONAL
      RETURN,0
    ENDIF

  ENDIF ELSE IF format EQ 'PITCH ANGLE vs ENERGY' THEN BEGIN

    IF NOT KEYWORD_SET(lb) THEN lb = [0]

    IF NOT KEYWORD_SET(ub) THEN ub = [input_header.nbands-1]

    ; extract data for detectors

    ies_extdata, { sttime:time_range(0),  entime:time_range(1), use_dets:detectors }, $
                 { use_bands:[1], nbands:1, botchan:0, topchan:0, lb:lb, ub:ub }, $
                   input_header, input_data, 0, header, data, errmsg, EPHEM=ephem, MF=mf, AT=at, MODES=modes

    IF errmsg NE '' THEN BEGIN
      IF NOT rcs_get_silent() THEN rcs_message, errmsg, /INFORMATIONAL
      RETURN,0
    ENDIF

  ENDIF

  ; change ctitle

  header.ztitle = 'Cts'

  RETURN, 1

END
