;+
; Project     :	CLUSTER II/RAPID
;
; Name        :	rcs_survey_panel
;
; Purpose     :	Produces panel for survey plots.
;
; Explanation : Produces panel for survey plots.
;
; Use         : < res = rcs_survey_panel(devname, file, npanels, pg_pos, sc, datatype, heading, $
;                                        start_time, end_time, detectors) >
;
; Inputs      : devname    : STRING standard device name.
;               file       : STRING giving output file name, needed externally for footnote
;               npanels    : INT no. of plot panels per page.
;               pg_pos     : INT panel number.
;               sc         : STRING characted giving S/C
;               datatype   : STRING IF data set type
;               heading    : STRING heading for plot
;               start_time : DOUBLE TAI start time of plot
;               end_time   : DOUBLE TAI end time of plot
;               detectors  : INTARR(input_header.ndetectors) flags indicating whether to use detectors or not.
;
; Opt. Inputs : None.
;
; Outputs     : res : INT flag indicating success of plot
;                       0 : failed
;                       1 : ok
;
; Opt. Outputs:	None.
;
; Keywords    : SETUP       = Flag indicating whether setup still required.
;               YTOP        = FLOAT position in page of top of panels.
;               YSIZ        = FLOAT size of panels.
;               XMARGIN     = FLTARR[2] xmargin size in characters.
;               YMARGIN     = FLTARR[2] ymargin size in characters.
;               /WINDOWS    = plots output in x window.
;               /PS         = plots output as postscript file
;                                if string then assumes it is a file name
;               /PNG        = plots output as PNG file
;                                if string then assumes it is a file name
;               /GIF        = plots output as GIF file
;                                if string then assumes it is a file name
;               /BWPS       = plots output as black and white postscript file
;                                if string then assumes it is a file name
;               RANGE       = FLTARR[2] count range.
;               ERANGE      = FLTARR[2] energy range.
;               FORMAT      = STRING indicating plot required.
;               XHEADER     = STRUCTURE header.
;               XDATA       = STRUCTURE data.
;               ELOG        = FLAG indicating logarithmic energy axis
;               LB          = INTARR lower bounds of energy channels
;               UB          = INTARR upper bounds of energy channels
;               CAL         = FLAG sets calibration on
;               DIV         = FLAG sets energy division on
;               RAT         = FLAG sets rates on
;               PROCNAME    = STRING overrides default procname.
;               AUX         = FLAG indicating should read auxiliary data
;               OCCUPANCY   = FLAG should perform OCCPANCY calculation.
;               POINTS      = FLAG indcating if lineplot requires points plotted
;               AXES        = INT indicating if LUT panel required
;               NOBAD       = FLAG indicating bad data not plotted
;               BINS        = FLAG indicating true energies shown (rather than channels)
;               NO_XAXIS    = FLAG controls whether x axis displayed in full plot
;               NO_END      = FLAG controls whether to finish page in full plot
;
; Written     :	Version 0.0, 12/08/02
;
; Modified    :	Version 0.1, 27/02/03
;                 Added AUX and OCCUPANCY keywords.
;                 Also LB, UB, PROCNAME, CAL, DIV, RAT, POINTS, AXES, NOBAD, BINS, NO_XAXIS, NO_END keywords.
;               Version 0.2, 03/03/03
;                 Moved LINEAR, BINS and ELOG keywords from rcs_survey_set to rcs_survey_calibrate.
;                 Added CHARSIZE keyword.
;               Version 0.3, 28/03/03
;                 Changed definition of logo positions.
;               Version 0.4, 24/09/03
;                 Changed initialization of draw widget block.
;
; Version     :	Version 0.4, 24/09/03
;-
;**********************************************************

FUNCTION rcs_survey_panel, devname, file, npanels, pg_pos, sc, datatype, heading, start_time, end_time, detectors, $
   SETUP=setup, YTOP=ytop, YSIZ=ysiz, XMARGIN=xmargin, YMARGIN=ymargin, $
   WINDOWS=windows, PS=postscript, PNG=png, GIF=gif, BWPS=bwps, $
   RANGE=range, ERANGE=erange, FORMAT=format, XHEADER=xheader, XDATA=xdata, ELOG=elog, $
   LB=lb, UB=ub, CAL=cal, DIV=div, RAT=rat, PROCNAME=procname, AUX=aux, OCCUPANCY=occupancy, $
   POINTS=points, AXES=axes, NOBAD=nobad, BINS=bins, NO_XAXIS=no_xaxis, NO_END=no_end, CHARSIZE=charsize

  COMMON ies_draw_widget_block, full_header, full_data, header, data, $
           output_info, output_chans, output_array, ids, dnfull, dnscroll, files, ephem, mf, at

  IF rcs_get_ifdata(sc, datatype, start_time, end_time, theader, tdata, /QUIET) THEN BEGIN

    ; transfer data to draw widget COMMON block

    rcs_draw_widget_block, theader, tdata, /SET

    ; extract data

    IF rcs_survey_extract ( [start_time, end_time], detectors, LB=lb, UB=ub ) THEN BEGIN

      ; process data

      IF rcs_survey_calibrate(0, CAL=cal, DIV=div, RAT=rat, BINS=bins, ELOG=elog, LINEAR=linear, PROCNAME=procname) THEN $
         RETURN, rcs_returns(devname, file, WINDOWS=windows, PS=postscript, PNG=png, $
                                     GIF=gif, BWPS=bwps, /UNFINISHED)

      ; set options

      rcs_survey_set, RANGE=range, AXES=axes, ERANGE=erange, PTITLE='SC'+sc, NOBAD=nobad, POINTS=points

      ; calculate occupancy

      IF KEYWORD_SET(occupancy) THEN rcs_calculate_occupancy

      ; plot data

      IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, $
                          GIF=gif, BWPS=bwps, PG_POS=pg_pos, NO_XAXIS=no_xaxis, NO_END=no_end, /FIXPANSIZ, NPANELS=npanels, $
                          HEADING = heading, $
                          FOOTNOTE = 'vs '+STRING(full_header.version) +' '+ies_time(FORM=9), $
                          SETUP=setup, YTOP=ytop, YSIZ=ysiz, XMARGIN=xmargin, YMARGIN=ymargin, CHARSIZE=charsize) THEN $
                  RETURN, rcs_returns(devname, file, WINDOWS=windows, PS=postscript, PNG=png, $
                                     GIF=gif, BWPS=bwps, /IOERROR)

      ; save counts for all detectors and energies

      rcs_draw_widget_block, xheader, xdata, /GET

    ENDIF

  ENDIF

  IF KEYWORD_SET(axes) THEN pg_pos = pg_pos + 2 ELSE pg_pos = pg_pos + 1

  ; leave info in sensible state

  output_info.minimum = -1
  output_info.maximum = -1

  RETURN, 1

END
