;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_PARSE_SCI
;
; Purpose     :	Loops through SCI file parsing lines.
;
; Explanation : Loops through SCI file parsing lines.
;               Version: MSF2IFF V5.0_20020607
;               Note that the first sample does not have the state fully defined.
;               The telemetry mode and lut are not set up until after the first K
;               code line.
;               However, subsequently have changes applying from that point on.
;               Have to do something special for the first sample.
;
; Use         : < res = rcs_parse_sci(unit) >
;
; Inputs      : unit   : LONG logical unti number for file
;
; Opt. Inputs : None.
;
; Outputs     : res : INT FLAG indicating success of operation.
;
; Opt. Outputs: None.
;
; Keywords    : EXPECTED : INTARR(4) no of samples expected in each telemetry mode
;
; Written     :	Version 0.0, 01/08/02
;
; Modified    : Version 0.1, 17/07/03
;                 Changed to function to allow error return.
;                 Added ON_IOERROR.
;                 Changed to return on IO error.
;               Version 0.2, 31/07/03
;                 Corrected bug : rcs_sci_throwaway not function.
;               Version 0.3, 03/09/03
;                 Renamed sci_head.version sci_head.cd_version.
;               Version 0.4, 18/09/03
;                 Added parsing of fill value comment.
;
; Version     :	Version 0.4, 18/09/03
;-
;**********************************************************

FUNCTION rcs_parse_sci, unit, EXPECTED=expected

  COMMON rcs_sci_block, sci_head, sci_data, state, samples

  ON_IOERROR, ioerror

  ; set up local string for each line

  line = ''

  ; set up no. of samples read in block

  nsamples = 0

  ; read each line

  ok = 1

  WHILE (NOT EOF(unit)) AND ok DO BEGIN

    READF, unit, line

    ; check if letter code is 'K'
    ; ie start of a new sample

    IF rcs_strstart(line,'K') THEN BEGIN

      IF NOT rcs_read_sci_sample(unit, line, EXPECTED=expected) THEN RETURN, 0

      nsamples = nsamples + 1

    ENDIF ELSE BEGIN

      IF KEYWORD_SET(nsamples) THEN rcs_text, '> read '+rcs_strtrim(nsamples)+' samples'

      nsamples = 0

      ; show line

      rcs_text, line

      ; interpret letter code

      CASE STRMID(line,0,1) OF

        'E' : IF NOT rcs_sci_throwaway(unit, line) THEN RETURN, 0

        'F' : IF NOT rcs_sci_throwaway(unit, line) THEN RETURN, 0

        'G' : IF NOT rcs_sci_throwaway(unit, line) THEN RETURN, 0

        'H' : IF NOT rcs_sci_throwaway(unit, line) THEN RETURN, 0

        '!' : BEGIN

          IF rcs_strstart(line,'! File name: ', POS=pos) THEN BEGIN

            rcs_text,'> file_name='+STRMID(line,pos,20)

          ENDIF ELSE IF rcs_strstart(line,'! Data class: ', POS=pos) THEN BEGIN

            sci_head.data_class = STRMID(line,pos,20)

            rcs_text,'> data_class='+sci_head.data_class

          ENDIF ELSE IF rcs_strstart(line,'! Contents: ', POS=pos) THEN BEGIN

            sci_head.contents = STRMID(line,pos,20)

            rcs_text,'> contents='+sci_head.contents

          ENDIF ELSE IF rcs_strstart(line,'! Fill Value:', POS=pos) THEN BEGIN

            sci_head.fill_value = FLOAT(STRMID(line,pos,20))

            rcs_text,'> fill value='+rcs_strtrim(sci_head.fill_value)

          ENDIF ELSE IF rcs_strstart(line,'! Spacecraft:  ', POS=pos) THEN BEGIN

            rcs_text,'> sc='+STRMID(line,pos,1)

          ENDIF ELSE IF rcs_strstart(line,'! Format NM: DATA(', POS=pos) THEN BEGIN

            rcs_text,'> nm format='+rcs_strmerge(rcs_strsplit(line, '(', ')', ','),',',ST='[',EN=']')

          ENDIF ELSE IF rcs_strstart(line,'! Format BM: DATA(', POS=pos) THEN BEGIN

            rcs_text,'> bm format='+rcs_strmerge(rcs_strsplit(line, '(', ')', ','),',',ST='[',EN=']')

          ENDIF ELSE IF rcs_strstart(line,'! Source CD-ROM(s): ', POS=pos) THEN BEGIN

            source_files = rcs_strsplit(STRMID(line,pos,40), '', '', ',')

            rcs_text,'> source files ='+rcs_strmerge(source_files,',',ST='[',EN=']')

            sci_head.cd_version = STRLOWCASE(STRMID(source_files[0],10,1))

          ENDIF

        END

        'I' : BEGIN

          IF rcs_strfind(line,'IES_int_time: ', POS=pos) THEN BEGIN
            CASE STRMID(line,pos,2) OF
              '02' : state.lut = 0
              '05' : state.lut = 1
              '15' : state.lut = 2
              '50' : state.lut = 3
              ELSE : rcs_message,/IMPLEMENTATION
            ENDCASE
            rcs_text,'> lut='+rcs_strtrim(state.lut)
          ENDIF ELSE IF rcs_strfind(line,'Telemetry_Mode: ', POS=pos) THEN BEGIN
            CASE STRMID(line,pos,3) OF
              'NM'  : state.type = 0
              'BM1' : state.type = 1
              'BM2' : state.type = 2
              'BM3' : state.type = 3
             ELSE : rcs_message,/IMPLEMENTATION
            ENDCASE
            rcs_text,'> mode='+rcs_strtrim(state.type)
          ENDIF

        END

        'M' : BEGIN

          ; not sure how whether these effect telemtry mode dont believe so

          ;IF rcs_strfind(line,'RAP_NOM:', POS=pos) THEN BEGIN
          ;  state.type = 0
          ;  rcs_text,'> mode='+rcs_strtrim(state.type)
          ;ENDIF ELSE IF rcs_strfind(line,'RAP_STBY:', POS=pos) THEN BEGIN
          ;  state.type = -2
          ;  rcs_text,'> mode='+rcs_strtrim(state.type)
          ;ENDIF

        END

        'C' : BEGIN

          ; caveats comment

        END

        ELSE : rcs_message, /IMPLEMENTATION

      ENDCASE

    ENDELSE

  ENDWHILE

  ; close file

  FREE_LUN, unit

  RETURN, 1

ioerror : RETURN, rcs_sci_ioerror(unit)

END
