;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_READ_SCI_SAMPLE
;
; Purpose     :	Reads values for one sample from SCI file.
;
; Explanation : Reads values for one sample from SCI file.
;               Puts info on sample in next sci_data sample and updates no. of points
;               in sci_head. Puts data in samples array and updates state.
;
; Use         : < res = rcs_read_sci_sample( unit, line )>
;
; Inputs      : unit   : LONG logical unti number for file
;               line   : STRING current line
;
; Opt. Inputs : None.
;
; Outputs     : res : INT FLAG indicating success of operation.
;
; Opt. Outputs: None.
;
; Keywords    : EXPECTED : INTARR(4) no of samples expected in each telemetry mode
;
; Written     :	Version 0.0, 05/07/02
;
; Modified    : Version 0.1, 17/07/03
;                 Changed to function to allow error return.
;                 Changed to return on IO error.
;               Version 0.2, 18/09/03
;                 Re-instated variable sized sci_samples array because of performance implications.
;                 Used sci_data.flag as fill_value flag and checked samples for fill values.
;               Version 0.3, 17/10/03
;                 Increase default increment size of storage.
;
; Version     :	Version 0.3, 17/10/03
;-
;**********************************************************

FUNCTION rcs_read_sci_sample, unit, line, EXPECTED=expected

  COMMON rcs_sci_block, sci_head, sci_data, state, samples

  ON_IOERROR, ioerror

  ; letter code for raw counts for sample

  ; set current sample state

  vals = rcs_strsplit(line)

  nvals = LONG(vals[1])

  ; correct state if necessary

  rcs_sci_correct_type, state, nvals, line, EXPECTED=expected

  ; increment stored data

  sci_head.npoints = sci_head.npoints + 1

  ; check sufficient storage for no of samples

  IF sci_head.npoints GT N_ELEMENTS(sci_data) THEN rcs_message, /IMPLEMENTATION

  sci_data[sci_head.npoints-1].pos = state.pos

  sci_data[sci_head.npoints-1].nvals = nvals

  sci_data[sci_head.npoints-1].time = anytim2tai(vals[2])

  sci_data[sci_head.npoints-1].edb_no = rcs_get_edb_no(vals)

  ; NB type and lut will not be defined from SCI file for first sample
  ; In STBY mode type will not be defined.
  ; NB type may have already been corrected for first point

  sci_data[sci_head.npoints-1].type = state.type

  sci_data[sci_head.npoints-1].lut = state.lut

  ; increase samples array storage (add 4MB)

  WHILE state.pos+nvals GT N_ELEMENTS(samples) DO BEGIN

    samples=[TEMPORARY(samples),FLTARR(1024L*1024L)]

  ENDWHILE

  ; deal with first sample

  IF sci_head.npoints EQ 2 THEN BEGIN

    IF sci_data[0].type LT 0 THEN sci_data[0].type = state.type

    IF sci_data[0].lut LT 0 THEN sci_data[0].lut = state.lut

  ENDIF

  ; read values

  WHILE nvals GT 0 DO BEGIN

    READF, unit, line

    ; check if blank letter code for current line
    ; ie is part of a data sample

    IF NOT rcs_strstart(line,' ') THEN rcs_message, /IMPLEMENTATION

    vals = rcs_strsplit(line,/FLOAT)

    new_state_pos = state.pos + N_ELEMENTS(vals)

    samples[state.pos:new_state_pos-1] = vals

    state.pos = new_state_pos

    nvals = nvals - N_ELEMENTS(vals)

  ENDWHILE

  ; check sample is finished correctly

  IF nvals NE 0 THEN rcs_message, /IMPLEMENTATION

  ; check if sample has any fill values

  sci_data[sci_head.npoints-1].flag = rcs_sci_check_fill_values(sci_head.fill_value, $
    samples[sci_data[sci_head.npoints-1].pos:sci_data[sci_head.npoints-1].pos+sci_data[sci_head.npoints-1].nvals-1])

  RETURN, 1

ioerror : RETURN, rcs_sci_ioerror(unit)

END
