;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_SCI_PROCESS_E3DD
;
; Purpose     :	Reads and processes SCI IES data.
;
; Explanation : Reads and processes SCI IES data.
;
; Use         : < rcs_sci_process_e3dd, sc, day, use, mdata_sci_head, mdata_sci_data, mdata_sci_samples >
;
; Inputs      : use   : STRUCTURE giving info on what IF data sets to create
;               sc    : STRING character giving S/C
;               day   : STRARR containing IF day name for processing.
;               mdata_sci_head    : STRUCTURE mdata header information
;               mdata_sci_data    : STRUCTURE[] mdata info for each sample
;               mdata_sci_samples : FLOAT[] mdata data samples
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    :
;
; Written     :	Version 0.0, 04/07/02
;
; Modified    : Version 0.1, 04/12/02
;                 Added OVERRIDE to 1st printout
;               Version 0.2, 11/07/02
;                 Modified to allow gzipped SCI files.
;               Version 0.3, 03/09/03
;                 Changed to new version control.
;               Version 0.4, 16/09/03
;                 Modified to pass mdata header, data and samples through argument list instead of
;                 just mdata.
;                 Added new mdata check.
;                 Added check for fill values.
;               Version 0.5, 17/10/03
;                 Removed ped and e3dd arrays as soon as possible after use.
;                 Only need MDATA if CLEANBM or CLEANBM2 - updated checks.
;               Version 0.6, 29/11/03
;                 Renamed rcs e3dd to rcs do e3dd.
;               Version 0.7, 08/01/04
;                 Renamed rcs sw version.
;
; Version     : Version 0.7, 08/01/04
;-
;**********************************************************

PRO rcs_sci_process_e3dd, sc, day, use, mdata_sci_head, mdata_sci_data, mdata_sci_samples

  ; get IES data

  IF KEYWORD_SET(use.e3ddbm) OR KEYWORD_SET(use.e3dd) OR $
     KEYWORD_SET(use.clean) OR KEYWORD_SET(use.cleanbm) OR $
     KEYWORD_SET(use.cleanbm2) OR KEYWORD_SET(use.pedestal) OR $
     KEYWORD_SET(use.espct) THEN BEGIN

     ; check if need MDATA

     IF (use.cleanbm OR use.cleanbm2) THEN mdata_required = 1 ELSE mdata_required = 0

     ; check if have MDATA

     IF mdata_required AND NOT KEYWORD_SET(mdata_sci_head) THEN BEGIN

       rcs_text, 'NO MDATA FOUND', /OVERRIDE

       RETURN

     ENDIF

     ; get e3dd raw count data

     e3dd_name = rcs_sciname(sc,day,'E3DD','K','*')
     ped_name = rcs_sciname(sc,day,'PED','K','*')

     rcs_text, 'Reading file : '+e3dd_name, /OVERRIDE

     rcs_text, 'Reading file : '+ped_name, /OVERRIDE

     ; test all files exist

     IF ies_file_exists(e3dd_name) AND ies_file_exists(ped_name) THEN BEGIN

       IF rcs_read_sci( e3dd_name, e3dd_sci_head, e3dd_sci_data, e3dd_sci_samples, $
                        EXPECTED=[8*9, 16*8*9, 8*9, 16*8*9] ) THEN BEGIN

         IF rcs_read_sci( ped_name, ped_sci_head, ped_sci_data, ped_sci_samples, $
                        EXPECTED=[2*9, 16*4*9, 2*9, 16*4*9] ) THEN BEGIN

           rcs_text, 'Processing E3DD SCI data'

           ; check e3dd and ped are compatible

           IF rcs_sci_check(e3dd_sci_head, e3dd_sci_data, e3dd_sci_samples, $
              ped_sci_head, ped_sci_data, ped_sci_samples) THEN rcs_message, /IMPLEMENTATION

           IF mdata_required THEN BEGIN

             ; check e3dd and mdata are compatible

             IF rcs_sci_check(e3dd_sci_head, e3dd_sci_data, e3dd_sci_samples, $
                mdata_sci_head, mdata_sci_data, mdata_sci_samples) THEN rcs_message, /IMPLEMENTATION

             ; convert to mdata FLTARR(32,n) 16 m values + 16 msigns for each sample

             mdata = REFORM(mdata_sci_samples,32,mdata_sci_head.npoints)

           ENDIF

           ; get vs

           IF NOT KEYWORD_SET(e3dd_sci_head.cd_version) THEN rcs_message, /IMPLEMENTATION

           vs = rcs_if_version( SOURCE=rcs_get_source(), CD_VERSION=e3dd_sci_head.cd_version, SW_VERSION=ies_sw_version())

           ; get flags indicating fill values

           flags = e3dd_sci_data.flag OR ped_sci_data.flag

           IF mdata_required THEN flags = flags OR mdata_sci_data.flag

           ; get flags indicating fill values

           list = WHERE(NOT flags, count)

           ; set up no of points rejected due to fill values

           n_duff_points = e3dd_sci_head.npoints - count

            ; get list of ends of runs of telemetry type

           type_list = ies_runs(e3dd_sci_data.type)

           ; loop through runs of each lut

           pos = 0

           FOR l = 0L, N_ELEMENTS(type_list)-1 DO BEGIN

             ; get telemetry type, size and end position of run

             type = e3dd_sci_data[type_list[l]].type

             end_pos = type_list[l]

             nsamples = end_pos - pos + 1

             ; process run of data

             ; check telemetry type and size
             ; if telemetry mode not known then junk

             IF type GE 0 THEN BEGIN

               IF rcs_check_bm(type) THEN BEGIN

                 list = WHERE(e3dd_sci_data[pos:end_pos].nvals NE 8*9*16 $
                           OR ped_sci_data[pos:end_pos].nvals NE 4*9*16, count)

                 IF count GT 0 THEN rcs_message, /IMPLEMENTATION

               ENDIF ELSE BEGIN

                 list = WHERE(e3dd_sci_data[pos:end_pos].nvals NE 8*9 $
                           OR ped_sci_data[pos:end_pos].nvals NE 2*9, count)

                 IF count GT 0 THEN rcs_message, /IMPLEMENTATION

               ENDELSE

               ; get data
               ; NM = [8,9], BM = [8,9,16]

               e0 = e3dd_sci_data[pos].pos

               e1 = e3dd_sci_data[end_pos].pos+e3dd_sci_data[end_pos].nvals-1

               p0 = ped_sci_data[pos].pos

               p1 = ped_sci_data[end_pos].pos+ped_sci_data[end_pos].nvals-1

               IF rcs_check_bm(type) THEN BEGIN

                 e3dd = REFORM(e3dd_sci_samples[e0:e1],8,9,16,nsamples)

                 ped = REFORM(ped_sci_samples[p0:p1],4,9,16,nsamples)

                 ; combine with pedestal channels = [12,9,16,nsamples]
                 ; seems unlikely but this really is what IDL does

                 e3ddbm = [TEMPORARY(ped),TEMPORARY(e3dd)]

                 ; if only one sample then will have dropped last dimension
                 ; idl eh !?

                 e3ddbm = REFORM(e3ddbm,12,9,16,nsamples)

                 ; rotate to [sectors, detectors, energies, nsamples]

                 e3ddbm = TRANSPOSE(e3ddbm,[2,1,0,3])

                 ; loop through samples

                 FOR k = pos, end_pos DO BEGIN

                   ; check for fill values

                   IF NOT flags[k] THEN BEGIN

                     ; set up mvals array

                     IF mdata_required THEN mvals = rcs_reconstruct_mvals(mdata[*,k])

                     ; get lut_no
                     ; NB lut_no is not adjusted for LUT changes

                     lut_no = e3dd_sci_data[k].lut

                     ; set flag

                     flag = 0

                     ; get time

                     time = e3dd_sci_data[k].time

                     ; make IF sets
                     ; NB lut_no has not been adjusted for LUT changes

                     rcs_do_e3dd, use, 0, e3ddbm[*,*,*,k-pos], mvals, lut_no, flag, sc, vs, time

                   ENDIF

                 ENDFOR

               ENDIF ELSE BEGIN

                 e3dd = REFORM(e3dd_sci_samples[e0:e1],8,9,nsamples)

                 ; although data appears as 2 rows of 9 values these are actually [2,9] array

                 ped = REFORM(ped_sci_samples[p0:p1],2,9,nsamples)

                 ; add together Pat's last two channels

                 e3dd[4,*,*] = e3dd[4,*,*]+e3dd[5,*,*]

                 e3dd[5,*,*] = e3dd[6,*,*]+e3dd[7,*,*]

                 e3dd = e3dd[0:5,*,*]

                 ; combine with pedestal channels = [8,9,nsamples]

                 e3ddnm = [TEMPORARY(ped),TEMPORARY(e3dd)]

                 ; loop through samples

                 FOR k = pos, end_pos DO BEGIN

                   ; check for fill values

                   IF NOT flags[k] THEN BEGIN

                     ; set up mvals array

                     IF mdata_required THEN mvals = rcs_reconstruct_mvals(mdata[*,k])

                     ; get lut_no

                     lut_no = e3dd_sci_data[k].lut

                     ; set flag

                     flag = 0

                     ; get time

                     time = e3dd_sci_data[k].time

                     ; make IF sets

                     rcs_do_e3dd, use, e3ddnm[*,*,k-pos], 0, mvals, lut_no, flag, sc, vs, time

                   ENDIF

                 ENDFOR

               ENDELSE

             ENDIF

             ; increment position in array

             pos = end_pos + 1

           ENDFOR

           IF n_duff_points GT 0 THEN rcs_text, 'No. of points rejected containing fill values = '+rcs_strtrim(n_duff_points)

         ENDIF ELSE rcs_text, 'NO PED SCI data :'+ped_name, /OVERRIDE

       ENDIF ELSE rcs_text, 'NO E3DD SCI data :'+e3dd_name, /OVERRIDE

    ENDIF ELSE rcs_text, 'E3DD OR PED SCI file does not exist', /OVERRIDE

  ENDIF

END
