;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_SCI_PROCESS_EPAD
;
; Purpose     :	Reads and processes SCI EPAD data.
;
; Explanation : Reads and processes SCI EPAD data.
;
; Use         : < rcs_sci_process_epad, sc, day, use, mdata_sci_head, mdata_sci_data, mdata_sci_samples >
;
; Inputs      : use   : STRUCTURE giving info on what IF data sets to create
;               sc    : STRING character giving S/C
;               day   : STRARR containing IF day name for processing.
;               mdata_sci_head    : STRUCTURE mdata header information
;               mdata_sci_data    : STRUCTURE[] mdata info for each sample
;               mdata_sci_samples : FLOAT[] mdata data samples
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    :
;
; Written     :	Version 0.0, 06/08/02
;
; Modified    : Version 0.1, 04/12/02
;                 Added OVERRIDE to 1st printout.
;               Version 0.2, 11/07/02
;                 Modified to allow gzipped SCI files.
;               Version 0.3, 21/08/03
;                 Modified error messages.
;               Version 0.4, 02/09/03
;                 Removed duplicate error messages.
;                 Added printout.
;                 Changed to new version control.
;               Version 0.5, 16/09/03
;                 Modified to pass mdata header, data and samples through argument list instead of
;                 just mdata.
;                 Added new mdata check.
;                 Added check for fill values.
;                 Allowed MDATA-EPAD mismatch to pass with error messages ie 010304.
;               Version 0.6, 17/10/03
;                 Improved check for fill values. Added check on MDATA also.
;               Version 0.7, 21/10/03
;                 Added mflags to cure bug introduced above.
;               Version 0.8, 08/01/04
;                 Renamed rcs sw version.
;
; Version     : Version 0.8, 08/01/04
;-
;**********************************************************

PRO rcs_sci_process_epad, sc, day, use, mdata_sci_head, mdata_sci_data, mdata_sci_samples

  ; get EPAD data

  IF KEYWORD_SET(use.epad) OR KEYWORD_SET(use.epad2) THEN BEGIN

    ; get epad raw count data

    name = rcs_sciname(sc,day,'EPAD','K','*')

    rcs_text, 'Reading file : '+name, /OVERRIDE

    ; test all files exist

    IF ies_file_exists(name) AND KEYWORD_SET(mdata_sci_head) THEN BEGIN

      IF rcs_read_sci( name, epad_sci_head, epad_sci_data, epad_sci_samples) THEN BEGIN

        rcs_text, 'Processing EPAD SCI data'

        ; check e3dd and mdata are compatible

        err = rcs_sci_check(epad_sci_head, epad_sci_data, epad_sci_samples, mdata_sci_head, mdata_sci_data, mdata_sci_samples)

        mdata = mdata_sci_samples

        mflags = mdata_sci_data.flag

        IF KEYWORD_SET(err) THEN BEGIN

          ; according to Pat this shouldnt happen anymore

          rcs_message, 'EPAD-MDATA data mismatch - check SCI files are up to date', /CONTINUE

          IF err EQ 1 THEN BEGIN

            ; allow epad to change but mdata is used elsewhere

            sci_head = mdata_sci_head
            sci_data = mdata_sci_data

            rcs_sci_fix, epad_sci_head, epad_sci_data, epad_sci_samples, sci_head, sci_data, mdata

            mflags = sci_data.flag

            rcs_text, 'Samples in Common : '+rcs_strtrim(sci_head.npoints),/OVERRIDE

          ENDIF ELSE rcs_message, /IMPLEMENTATION

        ENDIF

        ; convert to mdata FLTARR(32,n) 16 m values + 16 msigns for each sample

        mdata = REFORM(mdata,32,N_ELEMENTS(mdata)/32)

        ; get vs

        IF NOT KEYWORD_SET(epad_sci_head.cd_version) THEN rcs_message, /IMPLEMENTATION

        vs = rcs_if_version( SOURCE=rcs_get_source(), CD_VERSION=epad_sci_head.cd_version, SW_VERSION=ies_sw_version())

        ; set epad table

        epad_table = 1

        ; epad is fixed length

        epad = REFORM(TEMPORARY(epad_sci_samples),2,3,16,epad_sci_head.npoints)

        ; rotate to [sectors,detectors,energies,samples]

        epad = TRANSPOSE(epad,[2,1,0,3])

        ; get flags indicating fill values

        flags = epad_sci_data.flag OR mflags

        list = WHERE(NOT flags, count)

        ; set up no of points rejected due to fill values

        n_duff_points = epad_sci_head.npoints - count

        ; loop through samples

        FOR k = 0, count-1 DO BEGIN

          ; index into arrays

          kk = list[k]

          ; get lut_no

          lut_no = epad_sci_data[kk].lut

          ; get flag

          flag = 0

          ; get time

          time = epad_sci_data[kk].time

          ; set up mvals array

          mvals = rcs_reconstruct_mvals(mdata[*,kk])

          ; make IF sets

          rcs_epad, use, epad[*,*,*,kk], 0, mvals, lut_no, flag, sc, vs, time

        ENDFOR

        IF n_duff_points GT 0 THEN rcs_text, 'No. of points rejected containing fill values = '+rcs_strtrim(n_duff_points)

      ENDIF ELSE rcs_text, 'NO EPAD SCI data :'+name, /OVERRIDE

    ENDIF ELSE rcs_text, 'EPAD or MDATA SCI file does not exist', /OVERRIDE

  ENDIF

END
