;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	RCS_SCI_PROCESS_SGL01
;
; Purpose     : Reads and processes SGL01 and SGLBM SCI data.
;
; Explanation : Reads and processes SGL01 and SGLBM SCI data.
;
; Use         : < rcs_sci_process_sgl01, sc, day, use >
;
; Inputs      : use : STRUCTURE giving info on what IF data sets to cretae
;               sc  : STRING character giving S/C
;               day  : STRARR containing IF day name for processing.
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 23/07/02
;
; Modified    : Version 0.1, 04/12/02
;                 Added OVERRIDE to 1st printout.
;               Version 0.2, 11/07/02
;                 Modified to allow gzipped SCI files.
;               Version 0.3, 03/09/03
;                 Changed to new version control.
;               Version 0.4, 16/09/03
;                 Added check for fill values.
;               Version 0.5, 17/10/03
;                 Improved check for fill values.
;               Version 0.6, 08/01/04
;                 Renamed rcs sw version.
;
; Version     : Version 0.6, 08/01/04
;-
;**********************************************************

PRO rcs_sci_process_sgl01, sc, day, use

  ; global common block

  COMMON rcs_sgl01_block, header, data

  ; get SGL01 data

  IF KEYWORD_SET(use.sgl01) OR KEYWORD_SET(use.sglbm) THEN BEGIN

    ; get SGL1 raw count data

    name = rcs_sciname(sc,day,'SGL1','K','*')

    rcs_text, 'Reading file : '+name, /OVERRIDE

    ; test all files exist

    IF ies_file_exists(name) THEN BEGIN

      IF rcs_read_sci(name, sci_head, sci_data, sci_samples, $
                        EXPECTED=[7, 5*16, 7, 5*16] ) THEN BEGIN

        rcs_text, 'Processing SGL01 SCI data'

        ; get vs

        IF NOT KEYWORD_SET(sci_head.cd_version) THEN rcs_message, /IMPLEMENTATION

        vs = rcs_if_version( SOURCE=rcs_get_source(), CD_VERSION=sci_head.cd_version, SW_VERSION=ies_sw_version())

        ; restrict no .of messages

        bad_count = 0

        ; get flags indicating fill values

        list = WHERE(NOT sci_data.flag, count)

        ; set up no of points rejected due to fill values

        n_duff_points = sci_head.npoints - count

        ; loop through samples

        FOR k = 0, count-1 DO BEGIN

          ; index into arrays

          kk = list[k]

          ; check telemetry type and size

          IF sci_data[kk].type LT 0 THEN BEGIN

            ; dont use rcs_check_bm if type = -1

          ENDIF ELSE IF rcs_check_bm(sci_data[kk].type) THEN BEGIN

            IF sci_data[kk].nvals NE 5*16 THEN BEGIN

              sci_data[kk].type = -1

              bad_count = bad_count + 1

            ENDIF

          ENDIF ELSE BEGIN

            IF sci_data[kk].nvals NE 7 THEN BEGIN

              sci_data[kk].type = -1

              bad_count = bad_count + 1

            ENDIF

          ENDELSE

          ; if telemetry mode not known then junk

          IF sci_data[kk].type GE 0 THEN BEGIN

            ; get flag

            flag = 0

            ; get time

            time = sci_data[kk].time

            ; get data

            sgl01 = sci_samples[sci_data[kk].pos:sci_data[kk].pos+sci_data[kk].nvals-1]

            ; BM and NM produce two data sets, BM contributes to both

            IF rcs_check_bm(sci_data[kk].type) THEN BEGIN

              ; BM data

              ; reform into detectors = [ENY,TCR,TAC,STA,STO] vs sectors

              sgl01 = REFORM(sgl01,5,16)

              IF use.sglbm THEN BEGIN

                ; send to routine as [nsectors,ndetectors]

                rcs_add_sglbm, sc, vs, time, TRANSPOSE(sgl01), flag

              ENDIF

              IF use.sgl01 THEN BEGIN

                ; convert to ['ENY','TCR','TAC','STA0-7','ST00-7','STA8-15','STO8-15']

                sgl01 = [TOTAL(sgl01[0:2,*],2),REFORM(TOTAL(REFORM(sgl01[3:4,*],2,8,2),2),4)]

                IF NOT KEYWORD_SET(header) THEN ptitle = ['ENY','TCR','TAC','STA0-7','ST00-7','STA8-15','STO8-15']

                rcs_add_data, rcs_get_scname(sc)+'_IIMS_SGL01', sc, vs, 'SGL01', 'SGL01', 1, time, header, data, sgl01, 0, flag, 4.0, PTITLE=ptitle

              ENDIF

            ENDIF ELSE BEGIN

              ; NM data
              ; data in  ['ENY','TCR','TAC','STA0-7','STA8-15','ST00-7','STO8-15']
              ; data out ['ENY','TCR','TAC','STA0-7','ST00-7','STA8-15','STO8-15']

              sgl01[4:5] = sgl01[[5,4]]

              IF NOT KEYWORD_SET(header) THEN ptitle = ['ENY','TCR','TAC','STA0-7','ST00-7','STA8-15','STO8-15']

              rcs_add_data, rcs_get_scname(sc)+'_IIMS_SGL01', sc, vs, 'SGL01', 'SGL01', 1, time, header, data, sgl01, 0, flag, 16.0, PTITLE=ptitle

            ENDELSE

          ENDIF

        ENDFOR

        IF n_duff_points GT 0 THEN rcs_text, 'No. of points rejected containing fill values = '+rcs_strtrim(n_duff_points)

        IF bad_count GT 0 THEN rcs_message, 'SGL01 INCORRECT TELEMETRY TYPE ENCOUNTERED, COUNT= '+rcs_strtrim(bad_count),/INFORMATIONAL

      ENDIF ELSE rcs_text, 'NO SGL1 SCI data :'+name, /OVERRIDE

    ENDIF ELSE rcs_text, 'SCI file does not exist', /OVERRIDE

  ENDIF

END
