;+
; Project     :	POLAR - CEPPAD
;
; Name        :	eph_or_range
;
; Purpose     :	Finds orbit intervals from minimum L shell to threshold on north and south pole sides.
;               
; Explanation : Finds orbit intervals from minimum L shell to threshold on north and south pole sides.
;               Output list is time ordered.
;               Creates LSH_LIST_STRUCTURE array of orbit times.
;
; Use         : < xlsh = eph_or_range ( lstruct, lthreshold) >
;
; Inputs      : lstruct    : STRARR containing time and lshell values.
;               lthreshold : integer L shell threshold defining bands on north side
;
; Opt. Inputs : None.
;
; Outputs     : xlsh         : DBLARR(4,n) giving indices around L shell ranges and orbit type
;                              0 = north incoming
;                              1 = south incoming
;                              2 = south outgoing
;                              3 = north outgoing
;                              If invalid L shell values then returns -1. 
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	None.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None..
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/8/97
;
; Modified    :	Version 0.1, 10/9/98, MKC
;                 Used fixed south side threshold.       
;               Version 0.2, 11/5/99, MKC
;                 FIxed problem with a fake minimum found in data.       
;               Version 0.3, 14/1/02, MKC
;                 Added KEYWORD SHOW.       
;
; Version     :	Version 0.3, 14/1/02
;-
;**********************************************************

FUNCTION eph_or_range, lstruct, lthreshold, SHOW=show

  IF KEYWORD_SET(show) THEN WINDOW, 0

  str = ''

  nl = N_ELEMENTS(lstruct)

  ; get minima positions
  ; found one case where aroun l=30 got fake minimum

  list = WHERE ( lstruct[1:nl-2].lsh LT lstruct[0:nl-3].lsh AND $
                 lstruct[1:nl-2].lsh LE lstruct[2:nl-1].lsh AND $
                 lstruct[1:*].lsh LE 10, $
                 count)

  IF count LT 2 THEN MESSAGE,'Not sufficient MINIMA found'

  ; index list into lstruct

  list = 1+list

  ; start on even orbit

  IF lstruct[list[1]].time-lstruct[list[0]].time GT 8*3600. THEN list = list[1:*]

  ; end on odd orbit

  list = REFORM(list[0:(N_ELEMENTS(list)/2)*2-1],2,N_ELEMENTS(list)/2)

  ; check time differences

  dummy = WHERE ( (lstruct[list[1,*]].time - lstruct[list[0,*]].time) GT 8*3600., count)

  IF count GT 0 THEN MESSAGE, 'Invalid intra-orbit time differences found'

  dummy = WHERE ( (lstruct[list[0,1:*]].time - lstruct[list[0,0:(SIZE(list))(2)-2]].time) GT 19*3600. OR $
                  (lstruct[list[0,1:*]].time - lstruct[list[0,0:(SIZE(list))(2)-2]].time) LT 17*3600., count)

  IF count GT 0 THEN MESSAGE, 'Invalid inter-orbit time differences found'

  PRINT, 'Starting'

  ; loop through list

  dummy = {LSH_LIST_STRUCTURE_V3, time_start:0.0D0, time_end:0.0D0}
  struct = REPLICATE({LSH_LIST_STRUCTURE_V3}, 4, (SIZE(list))(2))

  FOR k = 0, (SIZE(list))(2)-1 DO BEGIN

    ; do incoming north

    pos = list[0,k]
    WHILE ( pos GT 0 AND lstruct[pos].lsh LT lthreshold ) DO pos = pos-1

    struct[0,k].time_start = lstruct[pos].time
    struct[0,k].time_end   = lstruct[list[0,k]].time

    in0 = pos
    in1 = list[0,k]

    ; do incoming south

    pos = list[0,k]
    WHILE ( pos LT nl-1 AND lstruct[pos].lsh LT lthreshold AND $
            lstruct[pos].lsh LT lstruct[pos+1].lsh ) DO pos = pos+1

    struct[1,k].time_start = lstruct[list[0,k]].time
    struct[1,k].time_end   = lstruct[pos].time

    is0 = list[0,k]
    is1 = pos

    ; do outgoing south

    pos = list[1,k]
    WHILE ( pos GT 0 AND lstruct[pos].lsh LT lthreshold AND $
            lstruct[pos].lsh LT lstruct[pos-1].lsh ) DO pos = pos-1

    struct[2,k].time_start = lstruct[pos].time
    struct[2,k].time_end   = lstruct[list[1,k]].time

    os0 = pos
    os1 = list[1,k]

    ; do odd north

    pos = list[1,k]
    WHILE ( pos LT nl-1 AND lstruct[pos].lsh LT lthreshold ) DO pos = pos+1

    struct[3,k].time_start = lstruct[list[1,k]].time
    struct[3,k].time_end   = lstruct[pos].time

    on0 = list[1,k]
    on1 = pos

    IF KEYWORD_SET(show) THEN BEGIN

      ; display data 
      ; NB if array has only one point still plot

      PLOT, lstruct[(in0-20)>0:(on1+20)<(nl-1)].time, $
            lstruct[(in0-20)>0:(on1+20)<(nl-1)].lsh, $
            yrange=[0,lthreshold+10]

      OPLOT, lstruct[in0:in1].time, lstruct[in0:in1].lsh, COLOR=200
      OPLOT, lstruct[is0:is1].time, lstruct[is0:is1].lsh, COLOR=150
      OPLOT, lstruct[os0:os1].time, lstruct[os0:os1].lsh, COLOR=100
      OPLOT, lstruct[on0:on1].time, lstruct[on0:on1].lsh, COLOR=50

      IF str NE 'g' AND str NE 'G' THEN BEGIN
        str = get_kbrd(1)
        IF str EQ 's' THEN STOP
      ENDIF

    ENDIF

  ENDFOR

  ; return structure

  RETURN, struct

END


