;+
; Project     :	POLAR - CEPPAD
;
; Name        :	get_inst_data
;
; Purpose     :	Gets instrument data.
;
; Explanation : Instrument data is available in daily files.
;
; Use         : < get_inst_data, time, inst, header, data, /QUIET >
;
; Inputs      : time       : time in any format.
;               inst       : STRING giving instrument name.
;
; Opt. Inputs : None.
;
; Outputs     : header : STRUCTURE giving data info
;               data   : STRUCTURE ARRAY containing data
;
; Opt. Outputs:	None.
;
; Keywords    : QUIET : If set suppresses printout.
;               REMOVE : Flag indicating temporary files to be removed.
;
; Calls       :	utc2tai, anytim2utc.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	Adapted from lsh_monitor.
;
; Written     :	Version 0.0, Martin Carter, RAL, 24/11/98
;
; Modified    :	Version 0.1, MKC, 22/3/99
;                 Allow MFE data.
;               Version 0.2, 19/5/00, MKC
;                 USe FIELD keyword to force MFE data.
;
; Version     :	Version 0.2, 19/5/00
;-
;**********************************************************

FUNCTION get_inst_data, time, inst, header, data, QUIET=quiet, REMOVE=remove

  ; check data required

  IF inst EQ 'MF' THEN BEGIN

    ; mf = get_mf(time,/TSY) ; gets field - TSY95
    mf = get_mf(time, /FIELD)

    IF NOT KEYWORD_SET(mf) THEN RETURN, 0

    ; set up a dummy IF structure

    nsamples = N_ELEMENTS(mf)

    header = ies_header_structure ('MF_V2', nsamples, 1, 1, 1, 1, $
                                    TITLE ='MF', $
                                    XTITLE = 'UT:', $
                                    YTITLE = 'Magnetic Field', $
                                    ZTITLE = 'nT', $
                                    PTITLE = ['BT'], $
                                    EXPERIMENT='MF' )

    data   = ies_data_structure ( nsamples, 1, 1, 1, EXPERIMENT='MF')

    header.tstart = mf[0].time
    header.tstop  = mf[nsamples-1].time
    header.ch_positions[0,*,0] = 0
    header.ch_positions[1,*,0] = 1

    data.time = mf.time
    data.data[0,0,0] = mf.bt

  ENDIF ELSE IF inst EQ 'MLAT' THEN BEGIN

    eph = get_ephem(time)

    IF NOT KEYWORD_SET(eph) THEN RETURN, 0

    ; set up a dummy IF structure

    nsamples = N_ELEMENTS(eph)

    header = ies_header_structure ('MLAT_V0', nsamples, 1, 1, 1, 1, $
                                    TITLE ='MLAT', $
                                    XTITLE = 'UT:', $
                                    YTITLE = 'Magnetic Latitude', $
                                    ZTITLE = 'deg', $
                                    PTITLE = ['MLAT'], $
                                    EXPERIMENT='MLAT' )

    data   = ies_data_structure ( nsamples, 1, 1, 1, EXPERIMENT='MLAT')

    header.tstart = eph[0].time
    header.tstop  = eph[nsamples-1].time
    header.ch_positions[0,*,0] = 0
    header.ch_positions[1,*,0] = 1

    data.time = eph.time
    data.data[0,0,0] = eph.mlat

  ENDIF ELSE BEGIN

    ; convert time to external time

    ext_time = anytim2utc ( time, /EXTERNAL )

    data_dir = concat_dir(ies_get_dir('IF',/POLAR),STRTRIM(STRING(ext_time.year),2))

    data_dir = concat_dir(data_dir,STRLOWCASE(inst))

    IF inst EQ 'IES' THEN mode = 'CLEAN_SURVEY' ELSE mode='SURVEY'

    file = concat_dir(data_dir,'PO_CEP_RAL_'+mode+'_'+inst+'_'+ $
           STRING( ext_time.year, ext_time.month, ext_time.day, FORMAT='(I4.4,I2.2,I2.2)' )+'.DAT*')

    ; find data file if present

    files = FINDFILE ( file, COUNT=count )

    IF count EQ 0 THEN BEGIN
      PRINT, 'File not found : '+file
      RETURN, 0
    ENDIF

    ; restore file

    ies_restore_gzip, files[0], header, data, /CHECK, REMOVE=remove

  ENDELSE

  RETURN, 1

END


