;+
; Project     :	CLUSTER - PEACE
;
; Name        :	LSH_GET_DATA
;
; Purpose     : Gets data for specified time.
;
; Explanation :	Gets data for specified time.
;               Returns 0 if unable to find data.
;
; Use         : < result = lsh_get_data(inst, detectors, times, REMOVE=remove)>
;
; Inputs      : inst      : STRING giving instrument type.
;               detectors : INTARR(ndetectors) flags indicating whether detectors should be used.
;               times     : DBLARR[*] giving TAI times of period of interest
;
; Opt. Inputs : None.
;
; Outputs     : Sets up common block variables :
;               input_header : structure giving data info.
;               input_data   : structure array containing data.
;               header : structure giving data info.
;               data   : structure array containing data.
;               ephem  : structure array containing ephemeris data.
;               at     : structure array containing attitude data.
;
; Opt. Outputs:	None.
;
; Keywords    : REMOVE    : flag indicating whether to leave data temporary data created.
;               MODES     : STRARR specifying HIST modes allowed.
;               EPH       : passes region of interest ephemeris data.
;
; Calls       : ies_concatenate, concat_dir, anytim2cal, anytim2utc, lsh_get_ephem.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 18/1198
;
; Modified    :	Version 0.1, MKC, 8/4/99
;                 Discovered bug which removed previous days ephem.
;               Version 0.2, 15/01/02
;                 Added keyword PA. Only need to get mf and at if pitch angle required.
;
; Version     :	Version 0.2, 15/01/02
;-
;**********************************************************

FUNCTION lsh_get_data, inst, detectors, times, REMOVE=remove, MODES=modes, EPHEM=eph, NFP=nfp, PA=pa

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information  ** used
  ; data          = structure array containing extracted data               ** used
  ; output_info   = structure containing plotting and processing info       ** used
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, input_header, input_data, header, data, $
           output_info, output_chans, output_array, ids, dnfull, dnscroll, files, ephem, mf, at

  ; explicitly set mf and at if pitch angle not required

  IF NOT KEYWORD_SET(pa) THEN BEGIN

    mf = 0

    at = 0

  ENDIF

  ; get start and end times

  start_time = times[0]
  end_time   = times[N_ELEMENTS(times)-1]

  ; check if current data exists

  renew = 1
  IF KEYWORD_SET(input_header) THEN BEGIN

    ; check if same instrument

    IF STRPOS(input_header.datatype,inst) GE 0 THEN BEGIN

      ; check if time overlap with current data

      IF start_time GE input_header.tstart AND start_time LE input_header.tstop THEN renew = 0

    ENDIF ; same instrument

  ENDIF ; no current data

  ; get data for start date

  IF renew THEN BEGIN

    ; get data

    IF NOT get_inst_data (start_time, inst, input_header, input_data, REMOVE=remove) THEN RETURN, 0

    IF KEYWORD_SET(pa) THEN BEGIN

      ; get attitude and mf data

      at    = get_at(start_time)

      mf    = get_mf(start_time, /MF)

    ENDIF

  ENDIF

  ; check if need more data
  ; need at most two days of data

  new = 0
  IF end_time GT input_header.tstop THEN BEGIN

    new = 1

    ; restore data for end date

    ; get data

    IF NOT get_inst_data (end_time, inst, new_header, new_data, REMOVE=remove) THEN RETURN, 0

    IF KEYWORD_SET(pa) THEN BEGIN

      ; get attitude data

      new_at    = get_at(end_time)

      new_mf    = get_mf(end_time, /MF)

    ENDIF

    ; concatenate data

    ies_concatenate, input_data, new_data, HEADER=input_header

    IF NOT KEYWORD_SET(at) THEN at = new_at $
    ELSE IF KEYWORD_SET(new_at) THEN BEGIN
      ies_concatenate, at, new_at, ERROR=error
      IF KEYWORD_SET(error) THEN MESSAGE,'Incompatible at data'
    ENDIF

    IF NOT KEYWORD_SET(mf) THEN mf = new_mf $
    ELSE IF KEYWORD_SET(new_mf) THEN BEGIN
      ies_concatenate, mf, new_mf, ERROR=error
      IF KEYWORD_SET(error) THEN MESSAGE,'Incompatible mf data'
    ENDIF

  ENDIF

  ; cut out relevant data

  ; get list of times within time interval

  time_list = WHERE ( input_data.time GT start_time AND input_data.time LT end_time, count )

  IF count EQ 0 THEN BEGIN
    PRINT, 'No data points in period '+anytim2cal(start_time,/DATE)+':'+anytim2cal(end_time,/DATE)
    RETURN, 0
  ENDIF

  ; grow time list

  start_pos = time_list[0]
  IF start_pos GT 0 THEN start_pos = start_pos - 1

  end_pos = time_list[count-1]
  IF end_pos LT input_header.npoints-1 THEN end_pos = end_pos + 1

  ; get region of interest ephemeris data

  eph = lsh_get_ephem( times, NFP=nfp)

  ; extract data for detectors

  ies_extdata, { sttime:input_data[start_pos].time,  entime:input_data[end_pos].time, use_dets:detectors }, $
               { use_bands:REPLICATE(1,input_header.nbands), nbands:0, $
                 botchan:0, topchan:input_header.nbands-1, $
                 lb:INDGEN(input_header.nbands), ub:INDGEN(input_header.nbands) }, $
                 input_header, input_data, 0, header, data, errmsg, EPHEM=eph, AT=at, MF=mf, MODES=modes

  IF errmsg NE '' THEN MESSAGE, errmsg

  ; NB do not carry over more than one day of data

  IF new THEN BEGIN

    input_header = new_header
    input_data   = new_data

    ; do not carry over more data than is needed outside

    IF KEYWORD_SET(at) THEN BEGIN

      list = WHERE(at.time GE start_time, count)

      IF count GT 0 THEN at = at[list]

    ENDIF

    ; check mf exists

    IF KEYWORD_SET(mf) THEN BEGIN
;    IF ies_tag_exists(mf,'TIME') THEN BEGIN ; ??? why
      list = WHERE(mf.time GE start_time, count)
      IF count GT 0 THEN mf = mf[list]
    ENDIF

  ENDIF

  RETURN, 1

END