;+
; Project     :	CLUSTER - PEACE
;
; Name        :	LSH_GET_EPHEM
;
; Purpose     : Gets ephem data for specified time.
;
; Explanation :	Gets data for specified time.
;               Returns 0 if unable to find data.
;
; Use         : < eph = lsh_get_ephem( times, REMOVE=remove)>
;
; Inputs      : times     : DBLARR[*] giving TAI times of period of interest
;
; Opt. Inputs : None.
;
; Outputs     : Sets up common block variables :
;               ephem  : structure array containing ephemeris data.
;               eph    : Returns ephemeris data in period of interest.
;
; Opt. Outputs:	None.
;
; Keywords    : REMOVE    : flag indicating whether to leave data temporary data created.
;
; Calls       : ies_concatenate, concat_dir, anytim2cal, anytim2utc
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	Created from lsh_ephem.
;
; Written     :	Version 0.0, Martin Carter, RAL, 10/06/99
;
; Modified    :	None
;
; Version     :	Version 0.0, 10/06/99
;-
;**********************************************************

FUNCTION lsh_get_ephem, times, NFP=nfp

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information  ** used
  ; data          = structure array containing extracted data               ** used
  ; output_info   = structure containing plotting and processing info       ** used
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, input_header, input_data, header, data, $
           output_info, output_chans, output_array, ids, dnfull, dnscroll, files, ephem, mf, at

  ; get start and end times

  start_time = times[0]
  end_time   = times[N_ELEMENTS(times)-1]

  ; check if current data exists

  renew = 1
  IF KEYWORD_SET(ephem) THEN BEGIN

    ; check if time overlap with current data

    IF start_time GE ephem[0].time AND start_time LE ephem[N_ELEMENTS(ephem)-1].time THEN renew = 0

  ENDIF ; no current data

  ; get data for start date

  IF renew THEN BEGIN

    ; get ephemeris data

    ephem = get_ephem(start_time, NFP=nfp)

  ENDIF

  ; check if need more data
  ; need at most two days of data

  IF end_time GT ephem[N_ELEMENTS(ephem)-1].time THEN BEGIN

    ; restore data for end date

    new_ephem = get_ephem(end_time, NFP=nfp)

    ; concatenate data

    IF NOT KEYWORD_SET(ephem) THEN ephem = new_ephem $
    ELSE IF KEYWORD_SET(new_ephem) THEN BEGIN
      ies_concatenate, ephem, new_ephem, ERROR=error
      IF KEYWORD_SET(error) THEN MESSAGE,'Incompatible ephem data'
    ENDIF

  ENDIF

  ; get list of times within time interval

  time_list = WHERE(ephem.time GE start_time AND ephem.time LE end_time, count)

  IF count EQ 0 THEN RETURN, -1

  ; do not carry over more data than is needed outside

  ephem = ephem[time_list[0]:*]

  RETURN, ephem[0:time_list[count-1]-time_list[0]]

END
