;+
; Project     :	CLUSTER - PEACE
;
; Name        :	PED_MONITOR
;
; Purpose     : Gets quiet pedestal position for each pedestal file and draws graph.
;
; Explanation :	Extract average value per orbit of pedestal position and temperature.
;
; Use         : <  >
;
; Inputs      :
;
; Opt. Inputs : None.
;
; Outputs     :
;
; Opt. Outputs:	None.
;
; Keywords    : TO : TAI time start
;               T1 : TAI time end
;               _EXTRA : PLOT keywords.

;
; Calls       :
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 22/4/97
;
; Modified    : Version 0.1, 02/10/03
;                 Changed ies get dir.
;                 Changed printout.
;
; Version     :	Version 0.1, 02/10/03
;-
;**********************************************************

PRO ped_monitor, times, peddat, ls, temperature, T0=t0, T1=t1, _EXTRA=extra

  ; come back out to top level

  ON_ERROR, 0 ; stay at error

  LOADCT, 3

  IF N_ELEMENTS(times) LE 1 THEN BEGIN

    ; process times

    IF KEYWORD_SET(t0) THEN tai0 = utc2tai(anytim2utc (t0))

    IF KEYWORD_SET(t1) THEN tai1 = utc2tai(anytim2utc (t1))

    ; get pedestal data file names

    ; choose pedestal files

    IF NOT KEYWORD_SET(t0) AND NOT KEYWORD_SET(t1) THEN BEGIN

      pedestal_files = PICKFILE ( PATH=ies_get_dir('PD',/POLAR), FILTER='PO_CEP_RAL_CLEAN_SURVEY_IES_*.PEDESTALS*', /READ, $
                                      TITLE='Select input data file' )

      IF pedestal_files EQ '' THEN MESSAGE, 'No pedestal data files found'

    ENDIF ELSE pedestal_files = 'PO_CEP_RAL_CLEAN_SURVEY_IES_*.PEDESTALS*'

    pedestal_files = FINDFILE( concat_dir ( ies_get_dir('PD',/POLAR), strip_dirname(pedestal_files)), COUNT=count)

    IF count EQ 0 THEN MESSAGE, 'No pedestal data files found'

    peddat = FLTARR(2,9)
    times  = 0d0
    ls     = 0
    temperature = 0.0

    FOR f = 0, N_ELEMENTS(pedestal_files)-1 DO BEGIN

      ped_file = pedestal_files(f)

      ; get tai time

      year  = STRMID(ped_file,STRPOS(ped_file,'PO_CEP_RAL_CLEAN_SURVEY_IES_')+28,4)
      dayno = LONG(STRMID(ped_file,STRPOS(ped_file,'PO_CEP_RAL_CLEAN_SURVEY_IES_')+32,3))

      utc = anytim2utc (year+'/1/1')
      utc.mjd = utc.mjd + dayno - 1

      tai = utc2tai ( utc )

      ok = 1
      IF KEYWORD_SET(t0) THEN ok = ok AND tai GE tai0
      IF KEYWORD_SET(t1) THEN ok = ok AND tai LE tai1

      IF ok THEN BEGIN

        ; decompress if necessary

        IF ies_test_gzip(ped_file) THEN BEGIN
          ies_gunzip, ped_file
          gzip = 1
         ENDIF ELSE gzip = 0

        ; set restore variables

        time = 0
        luts = 0
        pedestal_info = 0

        ; restore data

        PRINT, 'Restoring ', ped_file

        RESTORE, ped_file

        ; remove tmp file

        IF gzip THEN SPAWN, 'rm ' + ped_file

        ; get data details

        npoints    = (SIZE(pedestal_info))(2)
        nsectors   = (SIZE(pedestal_info))(3)
        ndetectors = (SIZE(pedestal_info))(4)

        ; use LUT change to find region outside of orbits

        ends = UNIQ(luts)

        IF N_ELEMENTS(ends) GT 1 THEN BEGIN

          ; correct UNIQ so that get indices of last element in run

          IF ends(N_ELEMENTS(ends)-1) NE N_ELEMENTS(luts)-1 THEN ends = [ends,N_ELEMENTS(luts)-1]
          ends = [-1,ends]

          ; sizes is size of each run of a given LUT

          sizes = ends(1:*) - ends(0:N_ELEMENTS(ends)-2)

          list = WHERE (sizes GE 5*3600/96., count)

;    PLOT, luts, YRANGE=[0,36]
;    OPLOT, ends(list), REPLICATE(20,count), PSYM=4, /noclip
;    OPLOT, ends(list+1), REPLICATE(20,count), PSYM=2, /noclip

          ; midpoints is middle of run of LUTS greater than 5 hours
          ; should not allow same run of LUTS to be counted twice even if spans
          ; different days but may not be in precise centre of run of LUTS in that case

          midpoints = (ends(list)+ends(list+1))/2

          ; get average pedestal positions at midpoints

          FOR k =0, N_ELEMENTS(midpoints)-1 DO BEGIN

            ; get average pedestal shift and width

            pdat = REFORM(REBIN(pedestal_info(0:1,midpoints(k)-3600/96:midpoints(k)+3600/96,*,*), 2, 1, 1, ndetectors))

            ; only get pedestal shifts need to add pedestal position for each detector

            FOR d=0,ndetectors-1 DO BEGIN

              ; NB this is not up to date

              dummy = ies_get_samples( time(midpoints(k)), luts(midpoints(k)), d+1,dummy_0,dummy_1,pedestal_posn,pedestal_width,/CENTROID )

              pdat(0, d) = pdat(0,d) + pedestal_posn

            ENDFOR

            times  = [times, time(midpoints(k))]
            peddat = [[[peddat]], [[pdat]]]
            ls     = [ls,luts(midpoints(k))]
          ENDFOR

          ; find temperature file if present

          str  = 'IES_'
          strl = STRLEN(str)

          ; default hk data directory

          tmp_file = concat_dir(ies_get_dir('HK',/POLAR), 'PO_CEP_RAL_HK_' + STRMID(ped_file, STRPOS(ped_file, str)+strl,8) + '.DAT*')

          temp_file = FINDFILE( tmp_file, COUNT=count)

          IF count EQ 1 THEN BEGIN

            tmp_file = temp_file(0)

            ; decompress if necessary

            IF ies_test_gzip(tmp_file) THEN BEGIN
              ies_gunzip, tmp_file
              gzip = 1
            ENDIF ELSE gzip = 0

            ; set restore variables

            input_header = 0
            input_data   = 0

            ; restore data

            PRINT, 'Restoring ', tmp_file

            RESTORE, tmp_file

            ; remove tmp file

            IF gzip THEN SPAWN, 'rm ' + tmp_file

            ; extract average temperatures

            FOR k =0, N_ELEMENTS(midpoints)-1 DO BEGIN

              list = WHERE ( input_data.time GE time(midpoints(k)-3600/96) AND input_data.time LE time(midpoints(k)+3600/96), count)

              IF count GT 0 THEN $
                temperature = [temperature, TOTAL(input_data(list).data)/count] $
              ELSE $
                temperature = [temperature, 0]

            ENDFOR

          ENDIF ELSE BEGIN

            FOR k =0, N_ELEMENTS(midpoints)-1 DO temperature = [temperature, 0]

            PRINT, 'No temperature data file : ' + tmp_file

          ENDELSE

        ENDIF ; uniq luts

      ENDIF ; ok

    ENDFOR

    peddat = peddat(*,*,1:*)
    times  = times(1:*)
    ls = ls(1:*)
    temperature = temperature(1:*)

  ENDIF

  tstart = times(0)
  tend   = times(N_ELEMENTS(times)-1)
  !P.CHARSIZE = 1
  x_chsize = get_chsize( /X )
  ies_xticks, tstart, tend, 1.0/x_chsize, xticks, xminor, xtickname, xtickv
  index = 0
  ylo = -3
  yhi =  3
  PLOT, [0,1], XRANGE=[tstart, tend], YRANGE=[ylo, yhi], /NODATA, YTICKLEN=-x_chsize, XSTYLE=5, YSTYLE=9, XMARGIN = [8,8], $
     TITLE='IES pedestal position variation with time for each detector', YTITLE='Pedestal position [bins]', _EXTRA=extra

  AXIS, YAXIS=1, YSTYLE=1, YRANGE=[ylo*10, yhi*10], YTITLE='Temperature [C]', YTICKLEN=-x_chsize, COLOR=200

  ies_do_x_axis, xticks, xminor, xtickname, xtickv, ylo, XTITLE='Date'

  FOR k =0, N_ELEMENTS(peddat(0,*,0))-1 DO OPLOT, times-times(0), peddat(index,k,*), _EXTRA=extra

  OPLOT,  times-times(0), temperature/10.0, COLOR=200, _EXTRA=extra

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

END