;+
; Project     :	CLUSTER - PEACE
;
; Name        :	READ_AP_DATA
;
; Purpose     : Reads Ap magnetic index data from NGDC.
;
; Explanation :	Data given as formatted file for whole year.
;
;               FORMAT FOR RECORDS OF SELECTED GEOMAGNETIC AND SOLAR ACTIVITY INDICES
;               -------------------------------------------------------------------------------
;               COLUMNS   FMT   DESCRIPTION
;               -------------------------------------------------------------------------------
;                1- 2     I2    YEAR
;                3- 4     I2    MONTH
;                5- 6     I2    DAY
;
;                7-10     I4    BARTELS SOLAR ROTATION NUMBER--a sequence of 27-day intervals
;                                 counted continuously from February 8, 1832.
;               11-12     I2    NUMBER OF DAY within the Bartels 27-day cycle.
;
;               13-14     I2    Kp or PLANETARY 3-HOUR RANGE INDEX for 0000 - 0300 UT.
;               15-16     I2    Kp or PLANETARY 3-HOUR RANGE INDEX for 0300 - 0600 UT.
;               17-18     I2    Kp or PLANETARY 3-HOUR RANGE INDEX for 0600 - 0900 UT.
;               19-20     I2    Kp or PLANETARY 3-HOUR RANGE INDEX for 0900 - 1200 UT.
;               21-22     I2    Kp or PLANETARY 3-HOUR RANGE INDEX for 1200 - 1500 UT.
;               23-24     I2    Kp or PLANETARY 3-HOUR RANGE INDEX for 1500 - 1800 UT.
;               25-26     I2    Kp or PLANETARY 3-HOUR RANGE INDEX for 1800 - 2100 UT.
;               27-28     I2    Kp or PLANETARY 3-HOUR RANGE INDEX for 2100 - 2400 UT.
;               29-31     I3    SUM of the eight Kp indices for the day expressed to the near-
;                                 est third of a unit.
;
;               32-34     I3    ap or PLANETARY EQUIVALENT AMPLITUDE for 0000 - 0300 UT.
;               35-37     I3    ap or PLANETARY EQUIVALENT AMPLITUDE for 0300 - 0600 UT.
;               38-40     I3    ap or PLANETARY EQUIVALENT AMPLITUDE for 0600 - 0900 UT.
;               41-43     I3    ap or PLANETARY EQUIVALENT AMPLITUDE for 0900 - 1200 UT.
;               44-46     I3    ap or PLANETARY EQUIVALENT AMPLITUDE for 1200 - 1500 UT.
;               47-49     I3    ap or PLANETARY EQUIVALENT AMPLITUDE for 1500 - 1800 UT.
;               50-52     I3    ap or PLANETARY EQUIVALENT AMPLITUDE for 1800 - 2100 UT.
;               53-55     I3    ap or PLANETARY EQUIVALENT AMPLITUDE for 2100 - 2400 UT.
;               56-58     I3    Ap or PLANETARY EQUIVALENT DAILY AMPLITUDE--the arithmetic mean
;                                 of the day's eight ap values.
;
;               59-61     F3.1  Cp or PLANETARY DAILY CHARACTER FIGURE--a qualitative estimate
;                                 of overall level of magnetic activity for the day determined
;                                 from the sum of the eight ap amplitudes.  Cp ranges, in steps
;                                 of one-tenth, from 0 (quiet) to 2.5 (highly disturbed).
;
;               62-62     I1    C9--a conversion of the 0-to-2.5 range of the Cp index to one
;                                 digit between 0 and 9.
;
;               63-65     I3    INTERNATIONAL SUNSPOT NUMBER.  Records contain the Zurich num-
;                                  ber through December 31, 1980, and the International Brus-
;                                  sels number thereafter.
;
;               66-70     F5.1  OTTAWA 10.7-CM SOLAR RADIO FLUX ADJUSTED TO 1 AU--measured at
;                                 1700 UT daily and expressed in units of 10 to the -22 Watts/
;                                 meter sq/hertz.  Observations began on February 14, 1947.
;                                 From that date through December 31, 1973, the fluxes given
;                                 here don't reflect the revisions Ottawa made in 1966.
;
;               71-71     I1    FLUX QUALIFIER.  "0" indicates flux required no adjustment;
;                                 "1" indicates flux required adjustment for burst in progress
;                                 at time of measurement; "2" indicates a flux approximated by
;                                 either interpolation or extrapolation; and "3" indicates no
;                                 observation.
;               -------------------------------------------------------------------------------
;
; Use         : < read_ap_data,  >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : .
;
; Opt. Outputs:	Produces plot.
;
; Keywords    : _EXTRA : PLOT keywords.
;
; Calls       : concat_dir, anytim2utc, utc2tai.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 23/07/98

;
; Modified    :	Version 0.1, MKC, RAL, 15/2/99
;                 Added HOURLY and DAILY keywords.
;               Version 0.2, 13/12/99
;                 Checked for Y2K compatability.
;               Version 0.3, 09/04/02
;                 Corrected for y2k.
;
; Version     :	Version .3, 09/04/02
;-
;**********************************************************

FUNCTION read_ap_data, _EXTRA=extra, DAILY=daily, HOURLY=hourly, HIGH=high

  ap_dir = GETENV('AP_DIR')

  IF ap_dir EQ '' THEN ap_dir = '/disk2/carter/plasma/ap'

  ; get Ap file names

  data_files = PICKFILE ( PATH=ap_dir, /READ, TITLE='Select input data files' )

  IF data_files EQ '' THEN MESSAGE, 'No data files found'

  data_files = FINDFILE( concat_dir ( ap_dir, strip_dirname(data_files)), COUNT=count)

  IF count EQ 0 THEN MESSAGE, 'No data files found'

  FOR f = 0, count-1 DO BEGIN

    ; open file

    OPENR, unit, data_files(f), /GET_LUN

    ; read records until eof encountered

    ; set up local string for each line

    temp_string = ''

    ; read each line

    WHILE (NOT EOF(unit)) DO BEGIN

      valid = 0

      ON_IOERROR, error

      READF, unit, temp_string

      IF N_ELEMENTS(array) EQ 0 THEN array =  temp_string $
        ELSE array = [ array, temp_string ]

      valid = 1

      error : IF NOT valid THEN PRINT, !ERR_STRING

    ENDWHILE

  ENDFOR

  ; close file

  FREE_LUN, unit

  ; process string array

    ; get dates in tai

    years = STRMID(array,0,2) 

    list0 = WHERE(STRPOS(years,'0') EQ 0, count0)

    list1 = WHERE(STRPOS(years,'0') NE 0, count1)

    IF count0 GT 0 THEN years[list0] = '20'+years[list0] 

    IF count1 GT 0 THEN years[list1] = '19'+years[list1]
 
    months = STRMID(array,2,2)

    days   = STRMID(array,4,2)

    dates = years + '/' + STRTRIM(months,1) + '/' + STRTRIM(days,1)

    tais = utc2tai(anytim2utc(dates))

    IF KEYWORD_SET(daily) THEN BEGIN

      ; get daily Ap index

      aps = FIX(STRMID(array,55,3))

    ENDIF ELSE BEGIN

      ; natural 3 hourly index

      nvals = N_ELEMENTS(tais)

      ; get hourly Dst index

      tais = REFORM(REPLICATE(1,8)#tais+FINDGEN(8)#REPLICATE(3*3600,nvals),8*nvals)

      aps = FLTARR(8,nvals)

      FOR k=0,7 DO aps[k,*] = FLOAT(STRMID(array,31+k*3,3))

      aps = REFORM(aps,8*nvals)

    ENDELSE


  ; set up necessary values for plot

  !P.CHARSIZE = 1
  !P.REGION = [ 0.0, 0, 1.0, 1]

  ; set up axes

  ies_do_axes, tais(0), tais(N_ELEMENTS(tais)-1), 0, 400, [8,2], [4,2], XTITLE='Time', YTITLE='Ap index'

  ; plot ap index

  oplot, tais-tais(0), aps

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  struct = REPLICATE({AP_STRUCTURE,time:tais[0],ap:aps[0]},N_ELEMENTS(aps))

  struct.time = tais
  struct.ap   = aps

  RETURN, struct

END