;+
; Project     :	CLUSTER - PEACE
;
; Name        :	READ_CUSP_DATA
;
; Purpose     : Reads Cusp crossings from Adam.  
;
; Explanation :	Data formatted as :
;               Date	        Time (UT)       Bz	By	Bx	Event		Comments
;
;               19/03/96	04:00 - 05:30	0	0	0	ED(d)		2 or 3 events
;                      		21:45 - 21:50	5	5	9	Cusp
;               012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789
;               0         1         2         3         4         5         6         7         8    
;
; Use         : < read_cusp_data >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : .     
;
; Opt. Outputs:	Produces plot.
;
; Keywords    : _EXTRA : PLOT keywords.
;
; Calls       : concat_dir, anytim2utc, utc2tai, get_utc.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 11/08/98

;
; Modified    :	
;
; Version     :	Version 0.0, 11/08/98
;-
;**********************************************************

PRO read_cusp_data, events, ranges, event_comments, range_comments, _EXTRA=extra


  IF N_ELEMENTS(events) LE 1 THEN BEGIN

  cusp_dir = GETENV('CUSP_DIR')

  IF cusp_dir EQ '' THEN cusp_dir = '/disk2/carter/plasma/cusps'

  ; get cusp data file name

  data_file = FINDFILE( concat_dir ( cusp_dir, 'cusps'), COUNT=count) 

  IF count EQ 0 THEN MESSAGE, 'No data file found'

  ; open file

  OPENR, unit, data_file[0], /GET_LUN

  ; read records until eof encountered

  ; set up local string for each line

  temp_string = ''  

  date = ''

  get_utc, utc
  tai = utc2tai(utc)
  events = tai
  ranges = [tai,tai]
  event_comments = ''
  range_comments = ''

  ON_IOERROR, error

  ; read data

  WHILE (NOT EOF(unit)) DO BEGIN

    valid = 0

    READF, unit, temp_string

;print, temp_string

    valid = 1

    ; separate line based on tabs

    temp_array = STR_SEP(temp_string,BYTE(9),/TRIM)

    ; separate tokens based on blanks

    t_array = '' 
    FOR k=0,N_ELEMENTS(temp_array)-1 DO t_array = [t_array,STR_SEP(temp_array[k],' ')]

    ; remove nulls

    list = WHERE(t_array NE '', count)
    IF count GT 0 THEN t_array=t_array[list]

    ; process line

;print,temp_array+'#'
;print,t_array+'#'

    ; look for time with :

    IF N_ELEMENTS(t_array) GE 4 THEN BEGIN

      IF STRPOS(t_array[0],'/') GE 0 THEN BEGIN

        date = t_array[0] 
        times = t_array[1]

        IF t_array[2] EQ '-' THEN BEGIN
          times = [times,t_array[3]]  
          bf   = t_array[4:6]
          IF N_ELEMENTS(t_array) GT 7 THEN evt  = t_array[7:*] ELSE evt = ''
        ENDIF ELSE BEGIN
          bf   = t_array[2:4]
          IF N_ELEMENTS(t_array) GT 5 THEN evt  = t_array[5:*]
        ENDELSE

      ENDIF ELSE IF STRPOS(t_array[0],':') GE 0 THEN BEGIN

        times = t_array[0]

        IF t_array[1] EQ '-' THEN BEGIN
          times = [times,t_array[2]]  
          bf   = t_array[3:5]
          IF N_ELEMENTS(t_array) GT 6 THEN evt  = t_array[6:*]
        ENDIF ELSE BEGIN
          bf   = t_array[1:3]
          IF N_ELEMENTS(t_array) GT 4 THEN evt  = t_array[4:*]
        ENDELSE

      ENDIF ELSE times = ''

      ; if valid data process it

      IF date NE '' AND times[0] NE '' THEN BEGIN

        ; get times and comments

        IF N_ELEMENTS(times) EQ 1 THEN BEGIN

          events = [events,utc2tai(anytim2utc(date+' '+times[0]))] 

          comment = ''
          FOR w = 0, N_ELEMENTS(evt)-1 DO comment = comment + ' ' + evt[w]
          event_comments = [event_comments, comment]

        ENDIF ELSE BEGIN

          ranges = [[ranges],[utc2tai(anytim2utc(date+' '+times[0])),utc2tai(anytim2utc(date+' '+times[1]))]]

          comment = ''
          FOR w = 0, N_ELEMENTS(evt)-1 DO comment = comment + ' '  + evt[w]
          range_comments = [range_comments, comment]

        ENDELSE

      ENDIF

    ENDIF

  ENDWHILE

  events = events[1:*]
  ranges = ranges[*,1:*]
  event_comments = event_comments[1:*]
  range_comments = range_comments[1:*]

error : IF NOT valid THEN MESSAGE, !ERR_STRING 

  ENDIF

  ; set up necessary values for plot

  !P.CHARSIZE = 1
  !P.REGION = [ 0.0, 0, 1.0, 1]

  ; set up axes 

  tstart = MIN([events[0],ranges[0,0]])

  nevents = N_ELEMENTS(events)
  nranges = N_ELEMENTS(ranges)/2

  ies_do_axes, tstart, MAX([events[nevents-1],ranges[1,nranges-1]]), 0, 5, [8,2], [4,2], XTITLE='Time', YTITLE='Event'

  ; plot event

  OPLOT, events-tstart, REPLICATE(1,nevents), PSYM = 4

  ; plot ranges

  FOR k = 0, nranges-1 DO PLOTS, ranges[*,k]-tstart, [2,2]

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

END 