;+
; Project     :	CLUSTER - PEACE
;
; Name        :	READ_DST_DATA
;
; Purpose     : Reads DST magnetic index data from NGDC.
;
; Explanation :	Data given as formatted file for whole year.
; Dst INDICES -------------------------------------------------------------------
; Dst (Disturbance Storm Time) equivalent equatorial magnetic disturbance indices
; are derived from hourly scalings of low-latitude horizontal magnetic variation.
; They show the effect of the globally symmetrical westward flowing high altitude
; equatorial ring current, which causes the "main phase" depression worldwide in
; the H-component field during large magnetic storms.  This diskette contains the
; hourly indices for the period 1 Jan 1957 through 30 Sep 1992, as derived by M.
; Sugiura and T. Kamei, WDC-C2 for Geomagnetism, Faculty of Science, Kyoto Uni-
; versity, Kyoto 606, Japan.
;
; Hourly H-component magnetic variations are analyzed to remove annual secular
; change trends from records of a worldwide array of low-latitude observatories.
; A cosine factor of the site latitude transforms residual variations to their
; equatorial equivalents and harmonic analysis isolates the term used as the
; Dst index.  Sugiura described Dst derivation in ANNALS OF THE IGY.
;
;                Dst INDICES RECORD FORMAT 1 Jan 1957 - 30 Sep 1992
; ===============================================================================
;  Column  Fmt  Description
; -------------------------------------------------------------------------------
;   1-  3   I3  Index Name or Station Code
;   4-  5   I2  Year (last 2 digits)
;   6-  7   I2  Month
;   8       A1  Geomagnetic Field Component H,D,Z,X,Y, or *=index, p=preliminary
;   9- 10   I2  Day
;  11- 16   I6  Reserved Area (all spaces)
;  17- 20   I4  Tabular or Base Value (100 nT units for H,Z,X,Y and degrees for D)
;  21- 24   I4  Value for  1st hour of day
;    .
;    .          H,Z,X,Y in units of 1 nT; D in units of 0.1 min
;    .
; 113-116   I4  Value for 24th hour of day
; 117-120   I4  Daily Mean Value
; -------------------------------------------------------------------------------
; To obtain absolute hourly value add base value to entry.
;               -------------------------------------------------------------------------------
;
; Use         : < dst=read_dst_data()  >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : .
;
; Opt. Outputs:	Produces plot.
;
; Keywords    : _EXTRA : PLOT keywords.
;
; Calls       : concat_dir, anytim2utc, utc2tai.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 23/07/98

;
; Modified    :	Version 0.1, MKC, RAL, 26/01/99
;                 Added HOURLY and DAILY keywords.
;               Version 0.2, 26/03/02
;                 Corrected for y2k.
;
; Version     :	Version .2, 26/03/02
;-
;**********************************************************

FUNCTION read_dst_data, _EXTRA=extra, DAILY=daily, HOURLY=hourly, HIGH=high

  dst_dir = GETENV('DST_DIR')

  IF dst_dir EQ '' THEN dst_dir = '/disk2/carter/plasma/dst'

  ; get Dst file names

  data_files = PICKFILE ( PATH=dst_dir, /READ, TITLE='Select input data files', GET_PATH=new_dst_dir )

  IF data_files EQ '' THEN MESSAGE, 'No data files found'

  data_files = FINDFILE( concat_dir ( new_dst_dir, strip_dirname(data_files)), COUNT=count)

  IF count EQ 0 THEN MESSAGE, 'No data files found'

  FOR f = 0, count-1 DO BEGIN

    ; open file

    OPENR, unit, data_files(f), /GET_LUN

    ; read records until eof encountered

    ; set up local string for each line

    temp_string = ''

    ; read each line

    WHILE (NOT EOF(unit)) DO BEGIN

      valid = 0

      ON_IOERROR, error

      READF, unit, temp_string

      IF N_ELEMENTS(array) EQ 0 THEN array =  temp_string $
        ELSE array = [ array, temp_string ]

      valid = 1

      error : IF NOT valid THEN PRINT, !ERR_STRING

    ENDWHILE

  ENDFOR

  ; close file

  FREE_LUN, unit

  ; process string array

  IF KEYWORD_SET(high) THEN BEGIN

    ; high time resolution data from Rumi

    array = array[1:*]

    ; get dates in tai

    years  = '19'+STRMID(array,0,2)
    list = WHERE(STRMID(array,0,3) EQ '190', count)
    IF count GT 0 THEN years[list]  = '20'+STRMID(years,2,2)

    months = STRMID(array,2,2)
    days   = STRMID(array,4,2)

    dates = years + '/' + STRTRIM(months,1) + '/' + STRTRIM(days,1)

    tais = utc2tai(anytim2utc(dates))

    tais = tais + FIX(STRMID(array,6,5))*60.

    ; get 5 minute average Dst SYMH index

    dsts = FLOAT(STRMID(array,65,9))

    ; get list of bad data

    list = WHERE(dsts EQ -999., count)

    IF count GT 0 THEN BEGIN

      PRINT, 'Bad data = ',count

      sub_dsts = FLOAT(STRMID(array,84,9))

      dsts[list] = sub_dsts[list]

    ENDIF

  ENDIF ELSE BEGIN

    ; NOAA hourly data

    ; get dates in tai

    years = STRMID(array,3,2) 

    list0 = WHERE(STRPOS(years,'0') EQ 0, count0)

    list1 = WHERE(STRPOS(years,'0') NE 0, count1)

    IF count0 GT 0 THEN years[list0] = '20'+years[list0] 

    IF count1 GT 0 THEN years[list1] = '19'+years[list1]
 
    months = STRMID(array,5,2)

    days   = STRMID(array,8,2)

    dates = years + '/' + STRTRIM(months,1) + '/' + STRTRIM(days,1)

    tais = utc2tai(anytim2utc(dates))

    IF KEYWORD_SET(hourly) THEN BEGIN

      nvals = N_ELEMENTS(tais)

      ; get hourly Dst index

      tais = REFORM(REPLICATE(1,24)#tais+FINDGEN(24)#REPLICATE(3600,nvals),24*nvals)

      dsts = FLTARR(24,nvals)

      FOR k=0,23 DO dsts[k,*] = FLOAT(STRMID(array,16,4))/100. + FLOAT(STRMID(array,20+k*4,4))

      dsts = REFORM(dsts,24*nvals)

    ENDIF ELSE BEGIN

      ; get daily average Dst index

      dsts = FLOAT(STRMID(array,16,4))/100.+FLOAT(STRMID(array,116,4))

    ENDELSE

  ENDELSE

  ; set up necessary values for plot

  !P.CHARSIZE = 1
  !P.REGION = [ 0.0, 0, 1.0, 1]

  ; set up axes

  ies_do_axes, tais(0), tais(N_ELEMENTS(tais)-1), -150, 50, [8,2], [4,2], XTITLE='Time', YTITLE='Dst index'

  ; plot Dst index

  OPLOT, tais-tais(0), dsts

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  struct = REPLICATE({DST_STRUCTURE,time:tais[0],dst:dsts[0]},N_ELEMENTS(dsts))

  struct.time = tais
  struct.dst  = dsts

  RETURN, struct

END