;+
; Project     :	CLUSTER - PEACE
;
; Name        :	READ_MFE_DATA
;
; Purpose     : Reads magnetic field data from MFE.
;
; Explanation :	Data given as formatted file for whole year.
;
;                 # NAME      UNITS     SOURCE                    FORMAT
;               001 UT        YR MON DY HR MN SC MS               6I3.2,I4.3
;               002 X_GSE     RE        POLAR_MFE                 G13.5
;               003 Y_GSE     RE        POLAR_MFE                 G13.5
;               004 Z_GSE     RE        POLAR_MFE                 G13.5
;               005 BX_GSE    nT        POLAR_MFE                 G13.5
;               006 BY_GSE    nT        POLAR_MFE                 G13.5
;               007 BZ_GSE    nT        POLAR_MFE                 G13.5
;               008 BT_GSE    nT        POLAR_MFE                 G13.5
;
; Use         : < read_mfe_data >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : .
;
; Opt. Outputs:	Produces plot.
;
; Keywords    : _EXTRA : PLOT keywords.
;
; Calls       : concat_dir, anytim2utc, utc2tai, get_utc.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 23/07/98

;
; Modified    :
;
; Version     :	Version 0.0, 23/07/98
;-
;**********************************************************

PRO read_mfe_data, _EXTRA=extra

  mfe_dir = GETENV('MFE_DIR')

  IF mfe_dir EQ '' THEN mfe_dir = '/disk2/carter/plasma/mfe'

  ; get mfe file names

  data_files = PICKFILE ( PATH=mfe_dir, /READ, TITLE='Select input data files' )

  IF data_files EQ '' THEN MESSAGE, 'No data files found'

  data_files = FINDFILE( concat_dir ( mfe_dir, strip_dirname(data_files)), COUNT=count)

  IF count EQ 0 THEN MESSAGE, 'No data files found'

  FOR f = 0, count-1 DO BEGIN

    ; open file

    OPENR, unit, data_files(f), /GET_LUN

    ; read records until eof encountered

    ; set up local string for each line

    temp_string = ''

    ON_IOERROR, error

    ; skip to NROWS

    WHILE (NOT EOF(unit)) AND STRMID(temp_string,0,7) NE 'NROWS =' DO BEGIN

      valid = 0

      READF, unit, temp_string

      PRINT, temp_string

      valid = 1

    ENDWHILE

    nrows = FIX(STRMID(temp_string,7,20))

    print,nrows

    ; skip to DATA:

    WHILE (NOT EOF(unit)) AND temp_string NE 'DATA:' DO BEGIN

      valid = 0

      READF, unit, temp_string

      PRINT, temp_string

      valid = 1

    ENDWHILE

    ; set up variables

    array = STRARR(nrows)

    ; read data

    FOR k = 0, nrows-1 DO BEGIN

      READF, unit, temp_string

      array(k) = temp_string

    ENDFOR
stop
    ; set up variables

    get_utc, temp_time, /EXTERNAL

    temp_data = FLTARR(7)

    ; read data

    WHILE (NOT EOF(unit)) DO BEGIN

      valid = 0

      READF, unit, temp_time, temp_data, FORMAT='(6I3.2,I4.3,07G13.5)'

      IF N_ELEMENTS(time_array) EQ 0 THEN time_array =  utc2tai(anytim2utc(temp_time)) $
        ELSE time_array = [ time_array, utc2tai(anytim2utc(temp_time)) ]

      IF N_ELEMENTS(data_array) EQ 0 THEN data_array =  temp_data $
        ELSE data_array = [ [data_array], [temp_data] ]

      valid = 1

    ENDWHILE

error : IF NOT valid THEN PRINT, !ERR_STRING

  ENDFOR

  ; close file

  FREE_LUN, unit
stop
  ; process string array

  ; get dates in tai

  years  = '19'+STRMID(array,0,2)
  list = WHERE(STRMID(array,0,3) EQ '190', count)
  IF count GT 0 THEN years[list]  = '20'+STRMID(years,2,2)

  months = STRMID(array,2,2)
  days   = STRMID(array,4,2)

  dates = years + '/' + STRTRIM(months,1) + '/' + STRTRIM(days,1)

  tais = utc2tai(anytim2utc(dates))

  ; get daily Ap index

  aps = FIX(STRMID(array,56,3))

  ; set up necessary values for plot

  !P.CHARSIZE = 1
  !P.REGION = [ 0.0, 0, 1.0, 1]

  ; set up axes

  ies_do_axes, tais(0), tais(N_ELEMENTS(tais)-1), 0, 400, [8,2], [4,2], XTITLE='Time', YTITLE='Ap index'

  ; plot ap index

  oplot, tais-tais(0), aps

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

END