;+
; Project     :	CLUSTER - PEACE
;
; Name        :	TMP_MONITOR
;
; Purpose     : Gets IES temperature for each HK file and draws graph.
;
; Explanation :	Extract minimum, maximum and average value per orbit of IES temperature.
;
; Use         : <  >
;
; Inputs      :
;
; Opt. Inputs : None.
;
; Outputs     :
;
; Opt. Outputs:	None.
;
; Keywords    : TO : TAI time start
;               T1 : TAI time end
;               _EXTRA : PLOT keywords.

;
; Calls       :
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 22/4/97
;
; Modified    : Version 0.1, 09/10/03
;                 Changed ies get dir.
;
; Version     :	Version 0.1, 09/10/03
;-
;**********************************************************

PRO tmp_monitor, struct, T0=t0, T1=t1, REMOVE=remove, _EXTRA=extra

  ; come back out to top level

  ON_ERROR, 1

  LOADCT, 3

  ; process times

  IF KEYWORD_SET(t0) THEN tai0 = utc2tai(anytim2utc (t0))

  IF KEYWORD_SET(t1) THEN tai1 = utc2tai(anytim2utc (t1))

  IF NOT KEYWORD_SET(struct) THEN BEGIN

    ; get data file names

    IF NOT KEYWORD_SET(t0) AND NOT KEYWORD_SET(t1) THEN BEGIN

      files = PICKFILE ( PATH=ies_get_dir('HK',/POLAR), FILTER='PO_CEP_RAL_HK_*.DAT*', /READ, $
                                      TITLE='Select input data file' )

      IF files EQ '' THEN MESSAGE, 'No data files found'

    ENDIF ELSE files = 'PO_CEP_RAL_HK_*.DAT*'

    files = FINDFILE( concat_dir ( ies_get_dir('HK',/POLAR), strip_dirname(files)), COUNT=count)

    IF count EQ 0 THEN MESSAGE, 'No data files found'

    struct = {TEMPERATURE__V0, time:0.d0, avr:0.0, min:0.0, max:0.0}

    FOR f = 0, N_ELEMENTS(files)-1 DO BEGIN

      tmp_file = files(f)

      ; get tai time

      year  = STRMID(tmp_file,STRPOS(tmp_file,'PO_CEP_RAL_HK_')+14,4)
      dayno = LONG(STRMID(tmp_file,STRPOS(tmp_file,'PO_CEP_RAL_HK_')+18,3))

      utc = anytim2utc (year+'/1/1')
      utc.mjd = utc.mjd + dayno - 1

      tai = utc2tai ( utc )

      ok = 1
      IF KEYWORD_SET(t0) THEN ok = ok AND tai GE tai0
      IF KEYWORD_SET(t1) THEN ok = ok AND tai LE tai1

      IF ok THEN BEGIN

        ; restore file

        ies_restore_gzip, tmp_file, input_header, input_data, /CHECK, REMOVE=remove

        ; extract temperatures

        list = WHERE ( input_data.data NE 0 AND input_data.data NE 255, count )

        IF count GT 0 THEN BEGIN

          input_data = input_data(list)

          struct = [struct, {TEMPERATURE__V0, time:(input_header.tstart+input_header.tstop)/2, $
                             avr:TOTAL(input_data.data)/count, $
                             min:MIN(input_data.data), $
                             max:MAX(input_data.data)} ]

        ENDIF

      ENDIF ; ok

    ENDFOR

    struct = struct[1:*]

  ENDIF

  IF NOT KEYWORD_SET(t0) THEN tai0 = struct[0].time
  IF NOT KEYWORD_SET(t1) THEN tai1 = struct[N_ELEMENTS(struct)-1].time

  list = WHERE ( struct.time GE tai0 AND struct.time LE tai1, count )

  IF count GT 0 THEN BEGIN

    min_temp = struct[list].min
    max_temp = struct[list].max
    avr_temp = struct[list].avr
    times    = struct[list].time

    ; CLEAN UP DATA

    count = N_ELEMENTS( times )

    IF count GT 1 THEN BEGIN

    list = WHERE ( ABS(min_temp(1:*) - min_temp(0:count-2)) GT 5, lcount )

      IF lcount GT 0 THEN FOR k = 0, lcount-1 DO min_temp(list(k)+1) = min_temp(list(k))

      list = WHERE ( ABS(max_temp(1:*) - max_temp(0:count-2)) GT 5, lcount )

      IF lcount GT 0 THEN FOR k = 0, lcount-1 DO max_temp(list(k)+1) = max_temp(list(k))

      list = WHERE ( ABS(avr_temp(1:*) - avr_temp(0:count-2)) GT 5, lcount )

      IF lcount GT 0 THEN FOR k = 0, lcount-1 DO avr_temp(list(k)+1) = avr_temp(list(k))

    ENDIF

    tstart = times(0)
    tend   = times(N_ELEMENTS(times)-1)
    !P.CHARSIZE = 1
    x_chsize = get_chsize( /X )
    ies_xticks, tstart, tend, 1.0/x_chsize, xticks, xminor, xtickname, xtickv
    index = 0
    ylo = 0
    yhi = 30
    PLOT, [0,1], XRANGE=[tstart, tend], YRANGE=[ylo, yhi], /NODATA, YTICKLEN=-x_chsize, XSTYLE=5, YSTYLE=1, XMARGIN = [8,8], $
     TITLE='IES average, min and max temperature variation with time', YTITLE='Temperature [C]', _EXTRA=extra

    ies_do_x_axis, xticks, xminor, xtickname, xtickv, ylo, XTITLE='Date'

    OPLOT,  times-times(0), avr_temp, _EXTRA=extra
    OPLOT,  times-times(0), min_temp, COLOR=200, _EXTRA=extra
    OPLOT,  times-times(0), max_temp, COLOR=200, _EXTRA=extra

    PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
           [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  ENDIF

END