;******************************************************************************
;* PROCEDURE:     
;*      p_polar_ceppad, panel, plotInfo, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written may 1996, Reiner Friedel
;******************************************************************************
pro p_polar_ceppad, panel, plotInfo, OUTPUT=OUTPUT, $
                OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
                SUBTABLE=SUBTABLE, _EXTRA=extra_par

COMMON plot_composer, widgetData
COMMON polar_eph, eph_header, eph_data 
COMMON polar_ceppad_data, input_header, input_data
COMMON polar_ceppad   ;see m_polar_cep_commons.pro for contents of this block
COMMON polar_ceppad_slice, time, yarr, zmat, data, $
                           extra_plotPar, utitle, uytitle, uztit
COMMON polar_ceppad_plot1, s1, s2, d1, d2, e1, e2, p1, p2, $
                           n_ch, n_sec, n_det, ndat, nodata

common mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
common yscale, yscl			;man/auto yscaling
common zscale, zscl			;man/auto zscaling
common shift_label, down                ;common for x-axis label shifting
common coordinateSystems, plotted_x, plotted_y  
                                       
COMMON papco_color_names
cols=[black, red, green,  blue, magenta, cyan, burgundy, $
      olive, dark_green, teal, royal_blue, violet]
  
forward_function polar_ceppad_flux_conv
  
control = *plotinfo.USR_PTR1

inst = strtrim(typeNames(control.product))
IF inst EQ 'IES' THEN mode = 'CLEAN' ELSE mode = 'SPIN'
av = varprt(control.sectors)+'Z'+varprt(control.spins)
IF av EQ '16Z16' THEN av = 'SURVEY' 
product = mode+'_'+av+'_'+inst

mask = control.mask
                
s1 = fix(control.SECTOR_AVERAGE_FROM_VAL)
s2 = fix(control.SECTOR_AVERAGE_TO_VAL)
d1 = fix(control.DETECTOR_AVERAGE_FROM_VAL)
d2 = fix(control.DETECTOR_AVERAGE_TO_VAL)
e1 = fix(control.ENERGY_AVERAGE_FROM_VAL)
e2 = fix(control.ENERGY_AVERAGE_TO_VAL)
p1 = fix(control.PITCH_AVERAGE_FROM_VAL)
p2 = fix(control.PITCH_AVERAGE_TO_VAL)
n_ch = input_header.NBANDS 
n_sec = input_header.NSECTORS 
n_det = input_header.NDETECTORS

if keyword_set(OUTPUT) then output=OUTPUT else output=0

; restrict data to actual time range requested - this makes for faster zooming
; input data is in TAI. For plotting, always start at zero to get maximum
; resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.time ge tai_1) AND (input_data.time le tai_2),c)
if c ne 0 then data=input_data(index) ELSE BEGIN
    msg = 'No data to plot' & GOTO, noplot
ENDELSE

max_t = max(data.time, max_t_idx)
data = data(0:max_t_idx)
input_header.NPOINTS = n_elements(data)

data.data = data.data*n_sec
xut1=0  &  xut2=tai_2-tai_1 & time = data.time-tai_1

uztit='['+input_header.ZTITLE+']'
utitle='POLAR CEPPAD!C'+product
ndat=n_elements(data)
nodata = 1e-20
   
ch1=transpose(input_header.CH_POSITIONS(0,*,0,0))
ch2=transpose(input_header.CH_POSITIONS(1,*,0,0))    
ch =  input_header.CH_POSITIONS

;IES channel conversion to keV: channel info is in ADC units - convert to KeV
pos = strpos(typeNames(control.product), 'IES')
IF pos NE -1 THEN BEGIN
    ch1 = ch1 * input_header.energy_cal(0) + input_header.energy_cal(1)
    ch2 = ch2 * input_header.energy_cal(0) + input_header.energy_cal(1)
ENDIF    

;check if any filtering or PSD is needed - get ephemeris/mag data first -------
filter = 0
FOR i = 0, n_elements(select_names)-1 DO BEGIN
    idx = where(tag_names(control) EQ select_names(i)+'_SELECT_RANGE') 
    IF control.(idx(0)) THEN filter = filter+1
ENDFOR 

IF ((filter GT 0) OR (control.cal EQ 3)) THEN BEGIN
    IF ((widgetData.need_to_draw EQ 1) AND (control.reload NE 1)) THEN BEGIN 
        get_polar_mag, control, mag_mfe, time
        get_polar_eph, control
    ENDIF 
ENDIF 

;make title strings for det, sec, pa and ek ranges
d1 = d1 >  0 <  (n_det-1) & d2 = d2 >  0 <  (n_det-1)
IF d1 EQ d2 THEN tit_det = 'Det '+varprt(d1) ELSE $
                 tit_det = 'Det '+varprt(d1)+'-'+varprt(d2)
s1 = s1 >  0 <  (n_sec-1) & s2 = s2 >  0 <  (n_sec-1) 
IF s1 EQ s2 THEN tit_sec = 'Sec '+varprt(s1) ELSE $
                 tit_sec = 'Sec '+varprt(s1)+'-'+varprt(s2)
e1 = e1 >  0 <  (n_ch-1) & e2 = e2 >  0 <  (n_ch-1) 
tit_ek = 'EK '+ strcompress(string(ch(0,e1,d1,0),ch(1,e2,d1,0),$
                 format = "(f6.1,'-',f6.1)"), /remove_all)
tit_pa = 'PA '+ varprt(p1)+'-'+varprt(p2)

valid = 0 & count=0 & nodata=1e-20 & ch_str = '' & zmat = 0

IF input_header.rat THEN uztit = uztit + '!Ns!E-1!N'
IF input_header.cal THEN uztit = uztit + '!Ncm!E-2!Nsr!E-1!N'
IF input_header.div THEN uztit = uztit + '!NkeV!E-1!N'

;select data to plot
CASE inst OF
    'IPS': BEGIN 
        n_spins = control.spins
        ;get detector assignment map      
        detector=input_header.DETECTORS
        data.data(*,detector,*)=data.data(*,*,*)
       ;correct efficiencies
        EFF= [3.5, 2.4, 2.0, 1.75, 1.5, 1.3, 1.2, 1.05, $
              1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0]
        m_polar_ceppad_zmat, control
        idx = where(zmat EQ nodata, c)
  
        CASE 1 OF 
            (control.pl_type EQ 0): BEGIN ;do Ek @ Det/Sec
                title = tit_det+', '+tit_sec
                utitle=utitle+'!C'+title
                uytitle=input_header.YTITLE+' (keV)'                
                yarr = fltarr(n_elements(ch1), 2)
                yarr(*, 0) = ch1 &  yarr(*, 1) = ch2
                zmat = zmat(*,0:n_ch-2)
                yarr = yarr(0:n_ch-2, *)
                FOR i = 0l, ndat-1 DO zmat(i, *) = zmat(i, *)*eff(0:15)
            END 
            (control.pl_type EQ 1): BEGIN ;do roll @ Ek/Det
                title = tit_ek + ', '+ tit_det
                utitle=utitle+'!C'+title
                yarr = fltarr(n_sec, 2)
                yarr(*, 0) = findgen(n_sec)+0.5
                yarr(*, 1) = yarr(*, 0)+1
                uytitle='sector #'
                valid=[1, 2, 4, 8]
                zmat = zmat*eff(e1)
            END  
            (control.pl_type EQ 3): BEGIN ;E @ Pa: Sum EK, choose PA
                title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                utitle=utitle+'!C'+title
                yarr = transpose(ch(*,*,d1,0))
            END
    
            (control.pl_type EQ 4): BEGIN ;do pitch at a Ek
                title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                utitle=utitle+'!C'+title
                uytitle = 'Pitch angle (deg.)'
                valid=[1, 2, 3, 4.5, 6, 9]
            END
        ENDCASE 
        
        IF c NE 0 THEN zmat(idx) = nodata

    END   

    'IES': BEGIN
        n_spins = control.spins
        ;limit energy channels according to e1,e2
        IF e1 GT 0 THEN data.data(*, *, 0:(e1-1)) = nodata
        IF e2 LT (n_ch-1) THEN data.data(*, *, (e2+1):(n_ch-1)) = nodata
        m_polar_ceppad_zmat, control

        CASE 1 OF 
            (control.pl_type EQ 0): BEGIN ;do Ek @ Det/Sec
                title = tit_det+', '+tit_sec
                utitle=utitle+'!C'+title
                uytitle=input_header.YTITLE+' (keV)'                
                yarr = fltarr(n_elements(ch1), 2)
                yarr(*, 0) = ch1 &  yarr(*, 1) = ch2

                IF control.cal EQ 3 THEN BEGIN ;fluxes at constant mu
                    ;make array of mu's
                    papco_m_mu_arr, control.mu, 100, 10000, 20, $
                                    mus, mus_str, ylow, yhigh
                    n_ch = n_elements(mus) & psd_mat = fltarr(ndat, n_ch)
                    ;may need to restrict mu calc by
                    ;taking out noisy or bad channels.
                    idx = [1, 2, 3, 4, 5, 6, 7, 8]
                    ;now call the mu / psd calc routine...
                    FOR i = 0, n_ch-1 DO BEGIN
                        papco_mu_calculation, yarr(idx, 0), yarr(idx, 1), $
                        mus(i), zmat(*, idx), mag_mfe(*, 3), PSD, /GEM  
                        psd_mat(*, i) = PSD
                    ENDFOR 

                    zmat = psd_mat
                    idx = where(finite(psd_mat) EQ 0, c)
                    IF c NE 0 THEN zmat(idx) = 1e-20

                    uztit = 'Hilmer PSD s!U3!N km!U-6!N'
                    uztit = 'Hilmer PSD c!u3!N/MeV!u3!Ncm!u3!N'
                    uytitle =  '!4l!3 (MeV/G)'
                    yarr = fltarr(n_ch, 2)
                    yarr(*, 0) = ylow & yarr(*, 1) = yhigh
                    input_header.rat = 0
                    input_header.cal = 0
                    input_header.div = 0
                ENDIF  
           
            END    

            (control.pl_type EQ 1): BEGIN ;do roll
                title = tit_ek + ', '+ tit_det
                utitle=utitle+'!C'+title
                yarr = fltarr(n_sec, 2)
                yarr(*, 0) = findgen(n_sec)+0.5
                yarr(*, 1) = yarr(*, 0)+1
                uytitle='sector #'
                valid=[1, 2, 4, 8]
            END  

            (control.pl_type EQ 3): BEGIN ;pitch E @ Pa: Sum EK, choose PArange
                yarr = transpose(ch(*,*,d1,0))
                title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                utitle=utitle+'!C'+title
            END
    
            (control.pl_type EQ 4): BEGIN ;do pitch at a Ek
                title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                utitle=utitle+'!C'+title
                uytitle = 'Pitch angle (deg.)'
                valid=[1, 2, 3, 4.5, 6, 9]
            END

            (control.pl_type EQ 5): BEGIN ;do mu range
                get_polar_mag, mag_bt, tai_1, tai_2, time
                zmat = fltarr(ndat, 2)
                ;****REM !!!!!!  treat relativistically 
                ;mu= <p2c2>/(B*2*Er)  where Er is rest energy
                ;<p2c2>= energy*(energy + (2*rest_energy))
                energy=[yminarr(0),ymaxarr(n_elements(ymaxarr)-2)]
                rest_energy=0.511*1000;keV
                for i=0,1 do begin
                    p2c2= energy(i)*(energy(i) + (2*rest_energy))
                    zmat(*,i)=p2c2/(mag_bt*rest_energy*2)
                endfor

                control.spec = 0 ;do as line plot
                idx = [0, n_elements(ymaxarr)-2]
                yminarr = yminarr(idx)
                ymaxarr = ymaxarr(idx)
                uytitle =  '!4l!3 MeV/G'
            END  
        ENDCASE    
    END  
  
    'HISTp SURVEY': BEGIN
        n_spins = 16
        zmat=fltarr(ndat,input_header.NBANDS)
        for i=0,ndat-1 do zmat(i,*)=data(i).DATA(sec,0,*)
        utitle=utitle+'!CS:' + string(sec,format='(i2.2)')
        uytitle=input_header.YTITLE+' (keV)'
    END 

    'HISTe SURVEY': BEGIN 
        n_spins = control.spins

        ;filter / limit data according to notes from Rich Selesnick.
        ;use only channels 2 through 14 in ABC mode and 
        ;10 through 14 in HBC mode
        valid_ych = intarr(16)

        ;select data for a given mode. if first 5 channels are zero,
        ;then mode is HBC, else it's ABC
        m_arr = lonarr(ndat)
        FOR i = 0l, ndat-1 DO m_arr(i) = total(data(i).data(*, 0, 0:5))

        CASE control.hist OF
            0: BEGIN            ;HBC mode
                valid_ych(10:14) = 1
                index = where(m_arr GT 0, c)
                IF c NE 0 THEN $  ;nodata the OTHER mode
                  FOR i = 0, c-1 DO data(index(i)).data(*, 0, *) = nodata
            END 
            1: BEGIN ;ABC mode
                valid_ych(2:14) = 1
                index = where(m_arr EQ 0, c)
                IF c NE 0 THEN $  ;nodata the OTHER mode
                  FOR i = 0, c-1 DO data(index(i)).data(*, 0, *)= nodata
            END 
        ENDCASE 
        idx = where(valid_ych EQ 0, c)
        IF c NE 0 THEN data.data(*, *, idx) = nodata

        m_polar_ceppad_zmat, control

        names = papco_make_choice_names(control.hist_info)
        utitle=utitle+'!C' + names(control.hist)

        CASE 1 OF 
            (control.pl_type EQ 0): BEGIN
                title = tit_det+', '+tit_sec
                utitle=utitle+'!C'+title
                uytitle=input_header.YTITLE+' (keV)'
                papco_2D_interplol, zmat, yarr, time, $
                    VALID_YCH = valid_ych, $
                    YCH_NUM = control.spec_intpol, NODATA = nodata
            END 
            (control.pl_type EQ 3): BEGIN ;pitch E @ Pa: Sum EK, choose PArange
                title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                utitle=utitle+'!C'+title
                yarr = transpose(ch(*,*,d1,0))
            END
            (control.pl_type EQ 4): BEGIN ;do pitch at a Ek
                title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                utitle=utitle+'!C'+title
                uytitle = 'Pitch angle (deg.)'
                valid=[1, 2, 3, 4.5, 6, 9]
            END 
            
        ENDCASE

    END 

    'HISTe': BEGIN    ;uncal counts. Use "new" Selesnick callibration
        ;16 sectors averaged over 4 spins, 32 sec resolution. Has pitch!
        n_spins = control.spins
        
        ;convert to counts per second
        out_info = {div:0, rat:1, cal:0, eff:0, subtract:0, shift:0}
        ies_docal, input_header, data, output_array, out_ch, nbands, out_info
        data.data = temporary(output_array)

        ;filter / limit data according to notes from Rich Selesnick.
        ;rates beyond 70 sample /sec are not valid - saturation
        ;temp = data.data
        ;idx = where(temp GT 70.0/n_sec, c)
        ;IF c NE 0 THEN temp(idx) = nodata
        ;data.data = temporary(temp)

        ;filter / limit data according to notes from Rich Selesnick.
        ;use only channels 2 through 14 in ABC mode and 
        ;10 through 14 in HBC mode
        valid_ych = intarr(16)
        
        ;select data for a given mode. if first 5 channels are zero,
        ;then mode is HBC, else it's ABC
        m_arr = lonarr(ndat)
        FOR i = 0l, ndat-1 DO m_arr(i) = total(data(i).data(*, 0, 0:5))

        CASE control.hist OF
            0: BEGIN            ;HBC mode
                valid_ych(10:14) = 1
                index = where(m_arr GT 0, c)
                IF c NE 0 THEN $  ;nodata the OTHER mode
                  FOR i = 0, c-1 DO data(index(i)).data(*, 0, *) = nodata
            END 
            1: BEGIN ;ABC mode
                valid_ych(2:14) = 1
                index = where(m_arr EQ 0, c)
                IF c NE 0 THEN $  ;nodata the OTHER mode
                  FOR i = 0, c-1 DO data(index(i)).data(*, 0, *)= nodata
            END 
        ENDCASE 
        idx = where(valid_ych EQ 0, c)
        IF c NE 0 THEN data.data(*, *, idx) = nodata

        IF control.cal EQ 1 THEN BEGIN ;get flux callibration
            ch = 1
            result = polar_ceppad_flux_conv(/HISTe, MODE=control.hist, $
                                            CHANNELS=ch)
            ;load new channel assignment  
            yarr = fltarr(16, 2)
            yarr(*, 0) = ch(0,*) &  yarr(*, 1) =  ch(1,*)
            input_header.CH_POSITIONS(0,*,0,0) = ch(0,*)
            input_header.CH_POSITIONS(1,*,0,0) = ch(1,*)
            input_header.cal=1 & input_header.div=1 & input_header.rat=1

            FOR i = 0l, ndat-1 DO BEGIN
                FOR s = 0, n_sec-1 DO BEGIN
                    spec = data(i).data(s, 0, *)
                    idx = where(spec NE nodata, c)
                    IF c NE 0 THEN data(i).data(s, 0, idx) = $
                                     data(i).data(s, 0, idx) * result(idx) 
                ENDFOR 
            ENDFOR   
        ENDIF

        m_polar_ceppad_zmat, control

        names = papco_make_choice_names(control.hist_info)
        utitle=utitle+'!C' + names(control.hist)

        CASE 1 OF 
            (control.pl_type EQ 0): BEGIN
                title = tit_det+', '+tit_sec
                utitle=utitle+'!C'+title
                uytitle=input_header.YTITLE+' (keV)'
                papco_2D_interplol, zmat, yarr, time, $
                    VALID_YCH = valid_ych, $
                    YCH_NUM = control.spec_intpol, NODATA = nodata
            END 
            (control.pl_type EQ 3): BEGIN ;pitch E @ Pa: Sum EK, choose PArange
                title = tit_pa + '!C'+ tit_det + ', ' + tit_sec
                utitle=utitle+'!C'+title
                yarr = transpose(ch(*,*,d1,0))
            END
            (control.pl_type EQ 4): BEGIN ;do pitch at a Ek
                title = tit_ek + '!C'+ tit_det + ', ' + tit_sec
                utitle=utitle+'!C'+title
                uytitle = 'Pitch angle (deg.)'
                valid=[1, 2, 3, 4.5, 6, 9]
            END 
            
        ENDCASE

    END 

    ELSE: BEGIN
        msg = 'plot product not supported'
        GOTO, noplot
    END  
ENDCASE 

;set all invalid data to "nodata"!!!
idx = where(data.flag EQ 0, c)
IF c NE 0 THEN zmat(idx, *) = nodata
    
;filter the data using the ephemeris/mag info ---------------------------------
;Handle different time bases - set non-filtered data to nodata flag.

;filter for B_X
IF control.B_X_SELECT_RANGE THEN BEGIN
    filt_var = mag_mfe(*, 0)
    v1 = control.B_X_SELECT_FROM_VAL
    v2 = control.B_X_SELECT_TO_VAL
    frmt = control.B_X_SELECT_RANGE_FRMT
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, format=frmt)
ENDIF

;filter for B_TOT
IF control.B_MAG_SELECT_RANGE THEN BEGIN
    filt_var = mag_mfe(*, 3)
    v1 = control.B_MAG_SELECT_FROM_VAL
    v2 = control.B_MAG_SELECT_TO_VAL
    frmt = control.B_MAG_SELECT_RANGE_FRMT
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, format=frmt)
ENDIF

;filter for model L
IF control.L_SELECT_RANGE THEN BEGIN
    filt_var = interpol(eph_data.MODEL_0_5.L, eph_data.tai-tai_1, time)
    v1 = control.L_SELECT_FROM_VAL
    v2 = control.L_SELECT_TO_VAL
    frmt = control.L_SELECT_RANGE_FRMT
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, format=frmt)
ENDIF

;filter for model MLAT
IF control.MLAT_SELECT_RANGE THEN BEGIN
    filt_var = interpol(eph_data.MODEL_0_5.MLAT, eph_data.tai-tai_1, time)
    v1 = control.MLAT_SELECT_FROM_VAL
    v2 = control.MLAT_SELECT_TO_VAL
    frmt = control.MLAT_SELECT_RANGE_FRMT
    idx = where((filt_var LE v1) OR (filt_var GE v2), c)
    IF c NE 0 THEN zmat(idx, *) = nodata
    utitle = utitle+'!C'+string(v1, v2, format=frmt)
ENDIF

;check if now any valid points are left
idx = where(zmat NE nodata, c)
IF c EQ 0 THEN BEGIN
    msg = 'No valid data left after filtering' & GOTO, noplot
ENDIF

yst = yarr(0, 0) & yen = yarr(n_elements(yarr(*, 0))-1, 1)

IF output EQ 2 THEN return

panelset,panel	               ;sets the panel position viewport

down=0

IF control.spec EQ 1 THEN BEGIN  ; do spec
    
    PAPCO_Set_SubTable, SUBTABLE ;sets the papco color table index

    IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
        message,'Using manual y-scale:'+varprt(yst)+' '+varprt(yen),/cont
    endif else begin
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
    ENDELSE 

    exclude = plotinfo.MANUALZSCALING

    if (zscl(panel(0),0) EQ 1) THEN BEGIN 
        zst=zscl(panel(0),1)  &  zen=zscl(panel(0),2)
    endif else begin
        papco_autorange,zmat,zst,zen, $
            log=zscl(panel(0),3), exclude=exclude, nodata=nodata
        zscl(panel(0),1)=zst  &  zscl(panel(0),2)=zen
    ENDELSE

    papco_y_label, yst, yen, log=yscl(panel(0),3), VALID = valid, /verb
    yst = yst*0.99 & yen = yen*1.01
   
    extra_plotPar={yrange:[yst, yen],xrange:[xut1, xut2], zrange:[zst,zen], $
               ylog:yscl(panel(0),3), zlog:zscl(panel(0),3),  $
               xtickformat:'noticks', $;ztitle:uztit, $
               ztickformat:'papco_color_bar_log_ticks'}

    ; add keyword structure set here with the one passed in
    extra_plotPar=create_struct(extra_plotPar, extra_par)

    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar   

    resx=control.spins*3 
    IF yscl(panel(0),3) THEN resy=(alog10(yen)-alog10(yst))/100 $
                        ELSE resy=(yen-yst)/100

    ; set all zero values to black!
    special_colors=fltarr(2,1)
    special_colors(0,0)=1e-20
    special_colors(1,0)=black    
    IF control.cal eq 4 THEN special_colors(1,0)=background

    papco_plot_colorspec,zmat,time,yarr, nodata = nodata, $
      resx=resx, resy=resy, $
      special_colors=special_colors, xstyle=5, ystyle=5, $
      _extra=extra_plotPar
    papco_color_bar,_EXTRA=extra_plotPar
    down=0
    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar   

    ;plot color bar label next to color bar.Use scalable routine!
    papco_colorbar_label, panel, uztit, /ROT90  
 
    ; plot y-axis label at left of plot. Use scalable routine!  
    left_side_label, panel, uytitle, /ROT90  

    right_side_label, panel, utitle, /ROT90, _extra=extra_plotpar

    ; store the coordinate information into common block
    plotted_x = !x  &  plotted_y = !y  
  
ENDIF ELSE BEGIN               ;do line
    
    IF (yscl(panel(0),0) EQ 1) THEN BEGIN 
        yst=yscl(panel(0),1)  &  yen=yscl(panel(0),2)
    ENDIF ELSE BEGIN
        papco_autorange, zmat, yst, yen, $
            log=yscl(panel(0),3), exclude=0, nodata=nodata        
        yscl(panel(0),1)=yst  &  yscl(panel(0),2)=yen
    ENDELSE 
    
    extra_plotPar={yrange:[yst,yen], xrange:[xut1, xut2], $
                   ylog:yscl(panel(0),3), xtickformat:'noticks'}    
    
    ; add keyword structure set here with the one passed in
    extra_plotPar=create_struct(extra_plotPar, extra_par)
    old_color = extra_plotPar.color

    papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar   
    
    ;utitle = utitle+'!C'+uytitle
    uytitle=uztit
    extra_plotPar.color=1

    plot, time, time, _extra=extra_plotPar,  xstyle=5, ystyle=5, /nodata
    plotted_x = !x  &  plotted_y = !y
    
    n=0 & count=0
   
    IF control.cal NE 3 THEN n_ch = (e2-e1)+1

    FOR i=0, n_ch-1 DO BEGIN
        IF n_ch EQ 1 THEN extra_plotPar.color = old_color $
          ELSE extra_plotPar.color=cols(i mod n_elements(cols))
        c=count/5 & mult=count mod 5
        IF (control.cal EQ 3) THEN name = mus_str(i) $
          ELSE name=string(round(yarr(i+e1,*)),format="(i4.4,'-',i4.4)")
        y=zmat(*,i) 
        index=where(y ne 1e-20,cc)
        if cc ne 0 then begin
            papco_gap_plot, time, y, nodata, AVERAGE = control.seg_av, $
              _extra=extra_plotpar,  /OPLOT
        ENDIF   
      
        IF n_ch NE 1 THEN BEGIN
            FOR j=1,c do name='!C'+name
            xyouts, !P.position(0)+0.01+0.135*mult, $
                    !P.position(3)-0.015, name, /normal, $
                    charsize=!p.charsize,color=cols(i mod n_elements(cols))
        ENDIF ;ELSE utitle=utitle+'!C'+name+' keV'
        count=count+1        
    ENDFOR   
    extra_plotPar.color = old_color

    ; plot y-axis label at left of plot. Use scalable routine!  
    IF NOT keyword_set(OVERPLOT) THEN left_side_label, panel, uytitle, /rot90

    papco_rs_lbl, panel, utitle, /LINE, _extra=extra_plotpar

ENDELSE

IF output EQ 1 THEN BEGIN
  message,'Writing plot data out to file', /cont
    description='POLAR CEPPAD - Detector/ channel/ sector data'
    time_label='Seconds since start of day'
    time_label='TAI'
time = time+tai_1
IF control.cal EQ 3 then begin
channels=mus
endif else begin
 channels=yarr
endelse

    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=nodata
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
ENDIF 

return 

;jump to here if no data to plot 
noplot: 
get_err_no = 10 & get_err_msg = msg
message, get_err_msg, /cont
papco_Draw_ErrorPanel, panel, plotinfo.panelkind+': ' + get_err_msg
plotted_x = !x  &  plotted_y = !y
  
END   

;******************************************************************************
; function to return the calibration matrix for CEPPAD.
; 
; Make conversion array of channles to flux for IPS
; Geometric factor = 2.78 e-3 for all channels 
; Need to use channel efficiencies too.
;   flux = conts s-1 * EFF * GF-1 * ECH-1    where ECH is the midpoint
;   energy.
; 
; CHANNELS - reports back calibrated energy channels for HIST
; MODE     - sets HIST mode
; PITCH    - indicates input array zmat is a pitch angle array 
;******************************************************************************
FUNCTION polar_ceppad_flux_conv, MODE=MODE, CHANNELS=CHANNELS, $
                                 IPS=IPS, HISTe=HISTe, HISTp=HISTp
  
COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON polar_ceppad ;see m_polar_cep_commons.pro for contents of this block

now=double(mjdt_start.mjd)+mjdt_start.t / 86400.0d ;decimal mjd
  
if keyword_set(IPS) then begin
    GF=2.78e-3  ;cm2 str 
    EFF= [3.5, 2.4, 2.0, 1.75, 1.5, 1.3, 1.2, 1.05, $
          1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0]
    emin=[ 16.8,  21.2,  27.9,  37.5,  49.6,  65.9,  87.7, 118.0, 161.0, $
          221.0, 303.0, 417.0, 574.0, 791.0,  1091.0, 1505.0  ]
    emax=[ 21.2,  27.9,  37.5,  49.6,  65.9,  87.7, 118.0, 161.0, 221.0, $
          303.0, 417.0, 574.0, 791.0, 1091.0, 1505.0, 2000.0  ]
    Ec= (emin+emax)/2
    conv_factor =  EFF / (GF * Ec)
    return,[conv_factor,1] ;add factor 1 for int. - stays in counts for IPS!
endif      
  
IF keyword_set(HISTe) THEN BEGIN
    area=1.98 
    GF=0.32 /2.0 ;cm2 str 
    Ew=fltarr(16,4)
    Ec=fltarr(16,4)
    ;mode:  0=HBC, 1=ABC, 2=ABC/HBC comb. 
    if keyword_set(MODE) then mode=MODE else mode=0

    ;enter updated values for energy channels, units of MeV
    ;Ew is width of enrgy channel, Ec is center of energy channel 

    ;ABC mode before 1997, day 24, hour20, index 0
    Ew(*,0)= [ .0039023, .0108258, .0327652, .0602893, $
               .1108010, .1712110, .2380950, .2550250, $
               .4397630, .4894990, .6232530, .7479280, $
               .9499060, 1.103000, 1.286780, .9417920  ]
    Ec(*,0)= [ 1.26624, .835552, .678460, .721913, $
               .861841, 1.06915, 1.29179, 1.56796, $
               1.90090, 2.44082, 3.13013, 4.03038, $
               5.12793, 6.43040, 8.06034, 10.1138  ]     
    ;HBC mode before 1997, day 24, hour20, index 1
    Ew(*,1)= [ .0039023, .0108258, .0327652, .0602893, $
               .1108010, .1712110, .2380950, .2550250, $ 
               .0433397, .0601939, .3833120, .640568, $
               .8741500, 1.089800, 1.251960, 1.15974  ]
    Ec(*,1)= [ .760273, .668676, .726457, .844293, $
               1.06937, 1.29300, 1.60456, 1.81422, $
               2.18184, 2.58126, 2.68224, 3.36205, $
               4.35261, 5.62261, 7.25908, 9.26742   ]    
    
    ;ABC mode after 1997, day 24, hour20, index 2
    Ew(*,2)= [ .0139147, .0328462, .0586733, .0849012, $
               .2198190, .1745390, .1036270, .3200350, $
               .3255060, .3796160, .4778030, .6588400, $
               .7839880, .9538890, 1.070490, 1.066160  ]
    Ec(*,2)= [ .760273, .668676, .726457, .844293, $
               1.06937, 1.29300, 1.60456, 1.81422, $
               2.18627, 2.67842, 3.28885, 3.96261, $
               4.87209, 5.97040, 7.20667, 8.86419  ]
    ;HBC mode after 1997, day 24, hour20, index 3
    Ew(*,3)= [ .0039023, .0108258, .0327652, .0602893, $
               .1108010, .1712110, .2380950, .2550250, $
               0.0433397, 0.0601939, 0.3833120, 0.640568, $
               0.8741500, 1.0898000, 1.2519600, 1.159740  ]
    Ec(*,3)= [ .760273, .668676, .726457, .844293, $
               1.06937, 1.29300, 1.60456, 1.81422, $
               2.18184, 2.58126, 2.68224, 3.36205, $
               4.35261, 5.62261, 7.25908, 9.26742   ]     
       
    ;get time for mode switch
    result=convert_date_to_t90(year=1997, doy=24, hour=20, /MJDT)
    tmch=double(result.mjd)+result.t / 86400.0d ;decimal mjd
    
    IF (now lt tmch) THEN BEGIN   ;before 1997, day 24, hour20
      ABC=0  &  HBC=1
    ENDIF ELSE BEGIN              ;after 1997, day 24, hour20
      ABC=2  &  HBC=3
    ENDELSE
    
    ;find conversion factors and channel assignment according to mode chosen
    CASE mode OF
      0:BEGIN ;HBC mode  
        conv_factor =  1.0 / (GF * Ec(*,HBC)*1000.) ;per keV
        if keyword_set(CHANNELS) then begin
            ch=fltarr(2,16)
            ch(0,*)=transpose((Ec(*,HBC) - Ew(*,HBC) / 2.0)*1000.0)
            ch(1,*)=transpose((Ec(*,HBC) + Ew(*,HBC) / 2.0)*1000.0)
            CHANNELS=ch
        ENDIF  
      END
      1:BEGIN ;ABC mode  
        conv_factor =  1.0 / (GF * Ec(*,ABC)*1000.) ;per keV
        if keyword_set(CHANNELS) then begin
            ch=fltarr(2,16)
            ch(0,*)=transpose((Ec(*,ABC) - Ew(*,ABC) / 2.0)*1000.0)
            ch(1,*)=transpose((Ec(*,ABC) + Ew(*,ABC) / 2.0)*1000.0)
            CHANNELS=ch
        ENDIF 
      END
      2:BEGIN ;HBC/ABC combination mode  
        conv_factor =  1.0 / (GF * Ec(*,ABC)*1000.) ;per keV
        if keyword_set(CHANNELS) then BEGIN ;in keV
            ch=fltarr(2,16)
            ch(0,0:9)  =transpose((Ec(0:9,ABC) - Ew(0:9,ABC) / 2.0)*1000.0)
            ch(1,0:9)  =transpose((Ec(0:9,ABC) + Ew(0:9,ABC) / 2.0)*1000.0)
            ch(0,10:15)=transpose((Ec(10:15,HBC) - Ew(10:15,HBC) / 2.0)*1000.0)
            ch(1,10:15)=transpose((Ec(10:15,HBC) + Ew(10:15,HBC) / 2.0)*1000.0)
            CHANNELS=ch
        ENDIF         
      END
    ENDCASE     
    return,[conv_factor]
ENDIF 
 
END 


;------------------------------------------------------------------------------
PRO get_polar_eph, control

COMMON polar_ephemeris
COMMON polar_eph, input_header, input_data 

new_plotinfo = papco_getplotinfostruct()
new_control = polar_eph_control

;set default mag field model 
new_control.L_CALC = 0
new_control.INTERNAL_MODEL = 0
new_control.EXTERNAL_MODEL = 5
new_control.ORIG = 1
new_plotinfo.USR_PTR1 = ptr_new(new_control)

r_polar_eph, new_plotinfo

END 


;******************************************************************************
;* FUNCTION: g_polar_ceppad_saturation
;*
;* DESCRIPTION:  returns the saturation levels if diff flux units
;*
;* KEYWORDS: INST - instrument sting (IPS, IES, HISTE HISTP
;*
;******************************************************************************
FUNCTION g_polar_ceppad_saturation,  INST = INST

IF NOT keyword_set(inst) THEN message, 'Need to specify keyword INST'

valid_inst = ['IPS', 'IES', 'HISTE', 'HISTP']

inst = strupcase(inst)

idx = where(valid_inst EQ inst, c)
IF c NE 1 THEN message, 'Not a valid instrument: '+inst

CASE inst OF
    'IPS'  : saturation = replicate(1e20,17) 
    'IES'  : saturation = replicate(1e20,16) 
    'HISTE': BEGIN
        saturation = replicate(1e20,16) 
        saturation[2:9] = [280, 300, 350, 200, 160, 130, 120, 70]
    END 
    'HISTP': saturation = replicate(1e20,16) 
END

return, saturation

END


;******************************************************************************
;* FUNCTION: g_polar_ceppad_background
;*
;* DESCRIPTION:  returns the background levels in flux
;*
;* INPUTS:   dec_y - time in decimal years
;*
;* KEYWORDS: INST - instrument sting (IPS, IES, HISTE HISTP
;*
;******************************************************************************
FUNCTION g_polar_ceppad_background, dec_y, INST = INST

IF NOT keyword_set(inst) THEN message, 'Need to specify keyword INST'

valid_inst = ['IPS', 'IES', 'HISTE', 'HISTP']

inst = strupcase(inst)

idx = where(valid_inst EQ inst, c)
IF c NE 1 THEN message, 'Not a valid instrument: '+inst

CASE inst OF 
    'IES'  : BEGIN 
        backgrd = replicate(0.0,10)
        backgrd(0) = 0.
        backgrd(1) = 15.
        backgrd(2) = $
          float(2.d0*(-6.52057651133840d-01*dec_y + 1.30747810544837d+03))
        backgrd(3) = $
          float(2.d0*(-2.87858835320972d-01*dec_y + 5.77250259052563d+02))
        backgrd(4) = $
          float(2.d0*(-1.33322512646315d-01*dec_y + 2.67401011454617d+02))
        backgrd(5) = $
          float(2.2d0*( -6.34251763143972d-02*dec_y + 1.27267399292487d+02))
        backgrd(6) = $
          float(2.2d0*(-3.40321956037584d-02*dec_y + 6.83332247132713d+01))
        backgrd(7) = $
          float(2.d0*(-2.62519632921723d-02*dec_y + 5.27654189697048d+01))
        backgrd(8) = $
          float(2.d0*( -2.97391652315982d-02*dec_y + 5.97140051701293d+01))
        backgrd(9) = $
          float(2.d0*(-7.52266086564119d-01*dec_y + 1.50931352772038d+03))
    END     
    'IPS'  : backgrd = replicate(0.0,17)
    'HISTE': BEGIN
        backgrd = replicate(0.0,16)
        backgrd(2)= 1.16e-2
        backgrd(3)= 1.e-2
        backgrd(4)= 8.e-3
        backgrd(5)= 8.e-3
        backgrd(6)= 6.5e-3
        backgrd(7)=float(2.5d0*(9.47951014960637d-05*dec_y^2 - $
          3.79324817741814d-01*dec_y + 3.79472145271080d+02))
        backgrd(8)=float(2.5d0*(1.08850529218508d-04*dec_y^2 - $
          4.35525009580665d-01*dec_y + 4.35650252964473d+02))
        backgrd(9)=float(2.7d0*(1.53963140600284d-04*dec_y^2 - $
          6.15839991806233d-01*dec_y + 6.15829342649824d+02))
        backgrd(10)=float(2.7d0*(2.27324595872668d-05*dec_y^3 - $ 
          1.36168923546231d-01*dec_y^2 + $
	  2.71886359978061d+02*dec_y - 1.80956701019402d+05))
        IF (dec_y LE 2001.20657551d0) THEN backgrd(11)=4.15e-3 ELSE $ 
          backgrd(11)=float(2.2d0*(-1.41219680963731d-03*dec_y^2 + $
              5.65803792921259d+00*dec_y - 5.66729608788766d+03))
        IF (dec_y LE 2000.96314417d0) THEN backgrd(12)=3.25e-3 ELSE $
          backgrd(12)=float(2.1d0*(7.58948885749372d-04*dec_y^3 - $
              4.56049570522622d+00*dec_y^2 + $
              9.13461493235652d+03*dec_y - 6.09883814040576d+06))
        IF (dec_y LE 2000.54015556d0) THEN backgrd(13)=3.5e-3 ELSE $
          backgrd(13)=float(2.1d0*(-2.88416710600342d-04*dec_y^4 + $
              2.31078042663128d+00*dec_y^3 - 6.94269596821599d+03*dec_y^2 + $
              9.27075351647305d+06*dec_y - 4.64229920353574d+09))
        backgrd(14)= 6.4e-3
    END 
    'HISTP': backgrd = replicate(0.0,16)
END

return, backgrd

END

;******************************************************************************
;* FUNCTION: g_polar_ceppad_crosscal
;*
;* DESCRIPTION:  scaling factors to cross compare ceppad with LANL GEO
;*               and GPS Flux corr = Flux/fcf
;*
;* KEYWORDS: INST - instrument sting (IPS, IES, HISTE HISTP
;*
;******************************************************************************
FUNCTION g_polar_ceppad_crosscal,  INST = INST

IF NOT keyword_set(inst) THEN message, 'Need to specify keyword INST'

valid_inst = ['IPS', 'IES', 'HISTE', 'HISTP']

inst = strupcase(inst)

idx = where(valid_inst EQ inst, c)
IF c NE 1 THEN message, 'Not a valid instrument: '+inst

CASE inst OF
    'IPS'  : crosscal = replicate(1.0,17) 
    'IES'  : crosscal = [1.0, 0.3, 0.2, 0.25, 0.25, 0.35, 0.5, 0.8, 1.2, 1.0]
    'HISTE': BEGIN
       crosscal = replicate(1.0,16)  
        crosscal[2:14] = [3, 2, 2.9, 1.8, 0.6, 1.4, 2, 2, 2, 2, 2, 2, 2]
    END 
    'HISTP': crosscal = replicate(1.0,16) 
END

return, crosscal

END


;******************************************************************************
;* PROCEDURE: m_polar_ceppad_zmat
;*
;* DESCRIPTION:  Make plot arrays for IES/IPS data. Also does Pitch!
;*
;******************************************************************************
PRO m_polar_ceppad_zmat, control

common polar_ceppad_data, input_header, input_data
common polar_ceppad         ;see m_polar_cep_commons.pro for definition
common polar_ceppad_slice   ;see pro p_polar_ceppad for definition
common mjdt, mjdt_start, mjdt_end ;common time limit in mjdt
COMMON get_error, get_err_no, get_err_msg  
COMMON polar_ceppad_plot1    ;see pro p_polar_ceppad for definition

;use mask of bad det/sec/ek to remove contamination
IF control.mask THEN BEGIN 
    COMMON polar_ceppad_ips_mask_data, mask_header, mask_data
    COMMON mjdt, mjdt_start, mjdt_end ;common time limit in mjdt

    f = data.data
    message, 'Masking contaminated data', /info
    r_polar_ceppad_ips_mask

    ;get mask onto same time base and sector base as data
    tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})

    data_time = time+tai_1
    mask_time = mask_data.tai

    ;Apply mask. look for mask closest to data time
    FOR i = 0l, n_elements(data_time)-1 DO BEGIN
        min = min(abs(mask_time-data_time(i)), min_idx)
        ;print,  min,  min_idx

        mask = rebin(transpose(mask_data(min_idx).data), n_sec, 9)
        idx1 = where(mask GT 0, c)
        IF c GT 0 THEN BEGIN
            d = f(*, *, *, i)
            FOR e = 0, n_ch-1 DO BEGIN
                dummy = d(*, *, e)
                dummy(idx1) = nodata
                d(*, *, e) = dummy
            ENDFOR 
            f(*, *, *, i) = d
        ENDIF 
    
    ENDFOR 

    ;cluster_rapid_ies_mask, control, mask, nodata_mask
    ;FOR i = 0l, ndat-1 DO f(*,*,*,i) = f(*,*,*,i)*mask + nodata_mask
    utitle = utitle+ '+ mask'
    data.data = temporary(f)
ENDIF

result = strsplit(typeNames(control.product), ' ', /extract)
inst = result(0)


;do adjustments for backgnd, satur, crosscal
add_utitle = ''
IF control.bckgnd THEN BEGIN
    message, 'Taking out background...', /info

    tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
    r = tai2utc(tai_1, /EXTERNAL)
    st_sec = UTC2TAI({year:r.year, MONTH:1, DAY:1, $
                  HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
    en_sec = UTC2TAI({year:r.year+1, MONTH:1, DAY:1, $
                  HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
    year_sec = en_sec-st_sec    
    dec_y = r.year+(tai_1-st_sec) / year_sec

    bckgnd = g_polar_ceppad_background(dec_y, INST = INST)

    tmp = data.data
    ;remember zero values - treat as no counts, not bad data
    idx_zero = where(tmp LE 0, c_zero)
    idx = where(tmp EQ nodata, c)
    FOR i = 0l, ndat-1 DO BEGIN 
        FOR d = d1, d2 DO BEGIN 
            FOR s = s1, s2 DO BEGIN
                tmp(s, d, *, i) = tmp(s, d, *, i) - bckgnd
            ENDFOR
        ENDFOR
    ENDFOR
    IF c NE 0 THEN tmp(idx) = nodata
    idx = where(tmp LE 0, c) & IF c NE 0 THEN tmp(idx) = nodata
    tmp(idx_zero) = 0
    data.data = tmp
    add_utitle = add_utitle+'b'
    
ENDIF

IF control.satur THEN BEGIN
    message, 'Taking out saturated values', /info

    satur = g_polar_ceppad_saturation(INST = INST)

    FOR d = d1, d2 DO BEGIN 
        FOR s = s1, s2 DO BEGIN
            FOR e = e1, e2 DO BEGIN
                one_e = data.data(s, d, e)
                idx = where(one_e GE satur(e), c)
                IF c NE 0 THEN one_e(idx) = nodata
                data.data(s, d, e) = one_e
            ENDFOR  
        ENDFOR  
    ENDFOR 
    add_utitle = add_utitle+',s'

ENDIF 

IF control.crosscal THEN BEGIN
    message, 'Doing spacecraft intercal', /info

    crosscal = g_polar_ceppad_crosscal(INST = INST)

    idx = where(tmp EQ nodata, c)
    FOR i = 0l, ndat-1 DO BEGIN 
        FOR d = d1, d2 DO BEGIN 
            FOR s = s1, s2 DO BEGIN
                tmp(s, d, *, i) = tmp(s, d, *, i) / crosscal
            ENDFOR
        ENDFOR
    ENDFOR
    IF c NE 0 THEN tmp(idx) = nodata
    add_utitle = add_utitle+',c'
ENDIF

IF add_utitle NE '' THEN add_utitle = '!U'+add_utitle+'!N'
utitle = utitle+add_utitle

CASE control.pl_type OF
    0: BEGIN    ;spectrum: average det, sec range chosen.
        zmat = fltarr(ndat, n_ch) & zmat(*) = 0
        numb = zmat & numb(*) = 0
        FOR i = 0l, ndat-1 DO BEGIN
            FOR d = d1, d2 DO BEGIN
                FOR s = s1, s2 DO BEGIN
                    x=where(data(i).data(s,d,*) NE nodata, c)
                    IF c NE 0 THEN BEGIN
                        zmat(i,x) = zmat(i,x)+data(i).data(s,d,x)
                        numb(i,x) = numb(i,x)+1
                    ENDIF 
                ENDFOR  
            ENDFOR  
        ENDFOR       
        x1 = where(numb NE 0, c1)
        IF c1 NE 0 THEN zmat(x1) =  zmat(x1) / numb(x1)
    END

    1: BEGIN     ;roll - sectors for a given det, Ek
        zmat =  fltarr(ndat, n_sec)
        FOR i = 0l, ndat-1 DO zmat(i, *) = data(i).data(*,d1,e1)
    END

    3: BEGIN    ;pitch E @ Pa: Sum energy, Pa range chosen.

        p = fix(data.pa*!RADEG)  ;get pitch angle array

        ;make data array - here average pitch angle range chosen
        ;need to preserve nodata flags
        f = data.data

        n_e = (e2-e1)+1
        zmat = fltarr(ndat, n_e) & numb = zmat
        d = 0

        message, 'Folding data into spectrum at pitch angle ranges '+ $
                  varprt(p1)+' to '+varprt(p2), /info

        IF n_det EQ 1 THEN BEGIN
            FOR i = 0l, ndat-1 DO BEGIN
                FOR s = s1, s2 DO BEGIN
                    pa = p(s, i)
                    IF (pa GE p1) AND (pa LE p2) THEN BEGIN
                        FOR e = e1, e2 DO BEGIN
                            IF f(s, d, e, i) NE nodata THEN BEGIN
                                zmat(i, e) = zmat(i, e) + f(s, d, e, i)
                                numb(i, e) = numb(i, e) + 1
                            ENDIF
                        ENDFOR 
                    ENDIF 
                ENDFOR
            ENDFOR             
        ENDIF ELSE BEGIN 
            FOR i = 0l, ndat-1 DO BEGIN
                FOR s = s1, s2 DO BEGIN
                    FOR d = d1, d2 DO BEGIN 
                        pa = p(s, d, i)
                        IF (pa GE p1) AND (pa LE p2) THEN BEGIN
                            FOR e = e1, e2 DO BEGIN
                                IF f(s, d, e, i) NE nodata THEN BEGIN
                                    zmat(i, e-e1) = zmat(i, e-e1) + f(s, d, e, i)
                                    numb(i, e-e1) = numb(i, e-e1) + 1
                                ENDIF
                            ENDFOR 
                        ENDIF 
                    ENDFOR
                ENDFOR
            ENDFOR 
        ENDELSE        

        x1 = where(numb EQ 0, c1, COMPLEMENT = x2, NCOMPLEMENT=c2)
        IF c2 NE 0 THEN zmat(x2) = zmat(x2) / numb(x2)
        IF c1 NE 0 THEN zmat(x1) = nodata

    END
    4: BEGIN    ;pitch Pa @ E: Average Pa, E range chosen.

        p = fix(data.pa*!RADEG)  ;get pitch angle array

        ;make data array - here sum energy range chosen
        ;since nodata is 1e-20, summing through it makes no differecne
        f = data.data(*, *, 0) & f(*) = 0

        FOR e = e1, e2 DO f = f + data.data(*, *, e)
        x = where(f LE (e2-e1+1)*nodata, c)
        IF c NE 0 THEN f(x) = nodata
       
        x = where((p LT 0) OR (f LE 0), c)
        IF c NE 0 THEN f(x) = nodata

        ;default pitch angle is pitch angle detector is looking at.
        ;if control.pa_bin is negative, sort into pitch angle particle
        ;is coming from!
        IF control.pa_bin LT 0 THEN BEGIN
            x = where(p NE nodata, c)
            p(x) = (180 - p(x)) MOD 180.0
            utitle = utitle+', from'
        ENDIF 

        yarr = papco_pitch_bin(abs(control.pa_bin),  PA_IDX = pa_IDX)
        n_ch = n_elements(yarr(*, 0))
        zmat = fltarr(ndat, n_ch) & numb = zmat

        IF control.cal EQ 4 THEN intercal = 1 ELSE intercal = 0

        IF NOT intercal THEN BEGIN

            ;fold data into pitch angle bins
            message, 'Folding data into Pitch angle ranges...', /info

            IF n_det EQ 1 THEN BEGIN
                FOR i = 0l, ndat-1 DO BEGIN
                    FOR s = s1, s2 DO BEGIN
                        IF f(s, i) NE nodata THEN BEGIN
                            pa = pa_idx(p(s, i))
                            zmat(i, pa) = zmat(i, pa) + f(s, i)
                            numb(i, pa) = numb(i, pa) + 1
                        ENDIF     
                    ENDFOR
                ENDFOR                
            ENDIF ELSE BEGIN     
                FOR i = 0l, ndat-1 DO BEGIN
                    FOR s = s1, s2 DO BEGIN
                        FOR d = d1, d2 DO BEGIN 
                            IF f(s, d, i) NE nodata THEN BEGIN
                                pa = pa_idx(p(s, d, i))
                                zmat(i, pa) = zmat(i, pa) + f(s, d, i)
                                numb(i, pa) = numb(i, pa) + 1
                            ENDIF     
                        ENDFOR
                    ENDFOR
                ENDFOR
            ENDELSE  

            x1 = where(numb EQ 0, c1, COMPLEMENT = x2, NCOMPLEMENT=c2)
            IF c2 NE 0 THEN zmat(x2) = zmat(x2) / numb(x2)
            IF c1 NE 0 THEN zmat(x1) = nodata

        ENDIF ELSE BEGIN

            ;fold data into pitch angle bins for d1
            message, 'Folding Det '+varprt(d1)+$
                     ' into Pitch angle ranges...', /info
            FOR i = 0l, ndat-1 DO BEGIN
                FOR s = 0, input_header.NSECTORS-1 DO BEGIN
                    FOR d = d1, d1 DO BEGIN 
                        IF (f(s, d, i) NE nodata) THEN BEGIN
                            pa = pa_idx(p(s, d, i))
                            zmat(i, pa) = zmat(i, pa) + f(s, d, i)
                            numb(i, pa) = numb(i, pa) + 1
                        ENDIF     
                    ENDFOR
                ENDFOR
            ENDFOR
            
            x1 = where(numb EQ 0, c1, COMPLEMENT = x2, NCOMPLEMENT=c2)
            IF c2 NE 0 THEN zmat(x2) = zmat(x2) / numb(x2)
            IF c1 NE 0 THEN zmat(x1) = nodata

            zmat1 = zmat
            zmat = fltarr(ndat, n_ch) & numb = zmat
            
            ;fold data into pitch angle bins for d2
            message, 'Folding Det '+varprt(d2)+$
                     ' into Pitch angle ranges...', /info
            FOR i = 0l, ndat-1 DO BEGIN
                FOR s = 0, input_header.NSECTORS-1 DO BEGIN
                    FOR d = d2, d2 DO BEGIN 
                        IF (f(s, d, i) NE nodata) THEN BEGIN
                            pa = pa_idx(p(s, d, i))
                            zmat(i, pa) = zmat(i, pa) + f(s, d, i)
                            numb(i, pa) = numb(i, pa) + 1
                        ENDIF     
                    ENDFOR
                ENDFOR
            ENDFOR

            x1 = where(numb EQ 0, c1, COMPLEMENT = x2, NCOMPLEMENT=c2)
            IF c2 NE 0 THEN zmat(x2) = zmat(x2) / numb(x2)
            IF c1 NE 0 THEN zmat(x1) = nodata
            
            zmat2 = zmat

            x = where((zmat1 NE nodata) AND (zmat2 NE nodata),  $
                          c, COMPLEMENT = x2)

            message, 'Finding ratio between det '+$
                     varprt(d1)+', '+ varprt(d2), /info
            IF c EQ 0 THEN BEGIN
                msg = 'No common pitch angles'
                GOTO, noplot
            ENDIF  
            zmat(x) = zmat1(x)/zmat2(x)
            zmat(x2) = nodata
            utitle=utitle+'!CRatio: '+varprt(d1)+'/'+varprt(d2)

            ;if average is set, and line plot, here average
            ;togther ratios from all pitch angles!
            IF (control.seg_av EQ 1) AND (control.spec EQ 0) THEN BEGIN
               message, 'Averaging all PA ratios',/info
                yray = fltarr(ndat) & yray(*) = nodata
                FOR i = 0l, ndat-1 DO BEGIN
                    y = zmat(i, *)
                    x1 = where(y NE nodata, c1)
                    y = alog10(y)
                    IF c1 NE 0 THEN BEGIN
                        min = min(y(x1), max = max)
                        ;IF (max-min) GT 0.5 THEN continue
                        yray(i) = 10^(total(y(x1))/c1)
                    ENDIF 
                ENDFOR     
                ;control.seg_av = 0
                n_ch = 1
                zmat = yray
                uztit = 'Av. all pitch'
            ENDIF

        ENDELSE ;do intercal

    END ;do PA @ Ek        

ENDCASE

idx = where(zmat LE 0, c)
IF c NE 0 THEN zmat(idx) = nodata

return

noplot:
stop

END 
