;******************************************************************************
;* file p_polar_ceppad_ips_mask.pro
;*
;* This file is a template for a plot routine which can be used by PAPCO.
;* It includes all the periphery needed. This routine makes use of a
;* color spectrogram routine that is supplied with PAPCO:
;* papco_XX/papco_lib/papco_plot_colorspec.pro. This routine should
;* meet all the requirements for the display of 3-D data (tyhe third
;* dimension being color). This routine also adheres to the PAPCO color scheme.
;* You of course can use your own color plot rotuine if desired!
;*
;* This is a working example for a pre-defined example data array.
;******************************************************************************
;******************************************************************************
;* PROCEDURE:     
;*      p_polar_ceppad_ips_mask, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           SUBTABLE=SUBTABLE, $
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type data_type for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*                additional axis (ie ephemeris) needed.
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar
;*              - panels all have common time
;*              - y-scaling (and/or z-scaling for color plots) is either
;*              - automatic or manually set using common yscale or zscale    
;*
;* INPUTS:       
;*	panel	three element integer vector. Controls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure PAPCD_PLOTINFO which is set
;*              by the panel editor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              PAPCO_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fields than these you have to make use of
;*              an extra common block.
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OUTPUT  if set call the papco routine that outputs plot data
;*              to a file.
;*    OVERPLOT  if you support this you can alter the appearance of your
;*              plot if it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field
;*              (see p_crres_mag for an example).
;* PLOTS_ATTHISPOSITION  used to control the left side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and line style.   
;* 
;* CALLING SEQUENCE:
;*       p_polar_ceppad_ips_mask, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written april 1996, Reiner Friedel
;*     added OUTPUT support, July 1998, , Reiner Friedel
;******************************************************************************
pro p_polar_ceppad_ips_mask, panel, type, swittch, OUTPUT=OUTPUT, $
                      OVERPLOT=OVERPLOT, $
                      PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
                      SUBTABLE=SUBTABLE, $
                      _EXTRA=extra_par
  
; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Line styles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,_EXTRA=extra_par
; are not used at all here, but are in the call for consistency...

; NOTE: I use a little trick in passing in the keywords to to this
; routine. The calling routine's call is simply:
;         p_polar_ceppad_ips_mask_line, panelVector, currPlotInfo.typeVector(0), $ 
;                           _EXTRA=extra_PlotInfo
; extra_PlotInfo contains the tags OVERPLOT, PLOTS_ATTHISPOSITION,
; SUBTABLE and then some tags which are meant to be passed to the plot
; call. By putting in the explicit keywords in the procedure definition they
; get split out of the structure extra_PlotInfo that was passed in, and only
; those tags which are meant for the plot routine remain in extra_par.

; Comments here should include an explanation of the plot options used
; in type and/or swittch. Replace type in the call with whatever variables
; you need, or add more. 

; The following common blocks are needed for PAPCO:
common mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
common yscale, yscl			;man/auto yscaling
common zscale, zscl			;man/auto zscaling
common shift_label, down             ;common for x-axis label shifting
common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords used by mouse

; One of the common blocks should contain the plot data as returned by the
; data read procedure r_polar_ceppad_ips_mask. 

   common polar_ceppad_ips_mask_data, input_header, input_data

   papco_Set_SubTable, SUBTABLE         ;sets the papco color table index
   panelset,panel			;sets the panel position viewport

; option to automatically calculate ytick numbers in order to avoid to
; many ticks if panels get too narrow. Maximum is set to 10. You may
; leave this out and allow plot to use the built-in defaults.
   ytickno=50*(!p.position(3)-!p.position(1))
   if ytickno gt 10 then ytickno=10
   ytickno=fix(ytickno) > 1
   
; here y-axis represents 32 sectors, always. So make sure ticks are
; multiples of 4
   if ytickno gt 4 then ytickno=(ytickno/4)*4

; since there is a common time axis plots look better if they all use the same
; number of xticks. In papco, this number is 6 by default. This is set by papco
; using the IDL graphics system variable !X.TICKS.
; You can overide this for your own plot by setting the varbale
; below. However, for plotting panels together, it makes sense to stick to the
; default. If you do set this variable, it needs to be included in the 
; extra_plotPar_common definition below.

;   xtickno=6

; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 
; Avoid setting the global IDL graphics system variables in your plot routine,
; as they may affect other modules' plot routines. The IDL graphics system 
; variables are set by papco to default values that gurantee some uniformity
; of plot appearance and should not be tampered with - unless you don't
; care if your plot "looks good" when combined with the other modules.
; NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! NB!!! 

; for the color plot routine, you need to set up arrays containing the upper  
; and lower limits of each channel. There must be as many entries as
; there are channels in the data array! 
n_channels=32
yarr=fltarr(n_channels,2)   

yarr(*,0)=findgen(32) & yarr(*,1)=yarr(*,0)+1

; the plot-calls normally use the arrays time for the time axis and yarr (zmat 
; for color plots) for the y or z axis.
; depending on your plot options different data may be selected and different
; axis-labeling may be needed. This is normally done using a case statement.

; in this example we have a single test array, so no case statement is needed.

; set the begin end end times. PAPCO convention is to use running secconds
; starting with the no of seconds since the beginning of the start
; day.
   time=input_data.time
   xut1=mjdt_start.t                       
   xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
   
   det=type(1)-1
   ndat=n_elements(input_data)
   zmat=fltarr(ndat,32)
   for i=0, ndat-1 do zmat(i,*)=input_data(i).data(det,*)
   
   uztit='mask'
   uytitle='sector'
   utitle='polar ceppad!Cips mask!CDet. '+strtrim(string(det+1),2)
   
   yst=yarr(0,0)
   yen=yarr(n_channels-1,1)

; restrict data to actual time range requested - this makes for faster zooming
   index=where((time ge xut1) and (time le xut2),count)
   if count ne 0 then begin
      zmat=zmat(index,*)
      time=time(index)
   endif
   
zmin = 0.0 & zmax = 2.0

; the variables used above have the following meaning:
;       uztit   : the z-axis (papco_colorbar) label
; 	uytitle : the y-axis label
;	utitle  : the right-of plot label defining the data

; the procedure papco_autorange (in papco_XX/papco_lib/autoscaling is
; optional and sets the zlimits automatically for the data, attempting
; to use the full dynamic range of the color table. you may need to
; supply your own procedure for your own data. Nodata sets the nodata
; flag used for the dataset.

; yscl or zscale is an array of dimensions (*,4) and contains info
; on the scaling to be used. The first dimension identifies the panel, and the
; next dimension has three entries:  
;         yscl(*,0)  is  0 for "use automatic scaling", 1 for "use manual"
;         yscl(*,1)  is then ymin (or zmin)
;         yscl(*,2)  is then ymax (or zmax)
;         yscl(*,3)  is 0 for lin, 1 for log axis
; when scaling is set to automatic, the automatic y limits are loaded in.
; for color plots, the z-limits are used for the color (z-axis) scaling  
   if (zscl(panel(0),0) eq 1) then begin
     zmin=zscl(panel(0),1)
     zmax=zscl(panel(0),2)
   endif else begin
     zscl(panel(0),1) = zmin
     zscl(panel(0),2) = zmax
   endelse

   if (yscl(panel(0),0) eq 1) then begin
     yst=yscl(panel(0),1)
     yen=yscl(panel(0),2)
   endif else begin
     yscl(panel(0),1)=yst
     yscl(panel(0),2)=yen
   endelse
   
; set up extra plot keywords: yrange, xrange, zrange  are always needed. 
; xtickformat is used to set the format of the x-axis ticks, by
; default  no ticks ar plotted. 
; For the rest you can use any of the kywords allowed with IDL's cplot
; routine. But beware that some keywords have default values set by
; PAPCO:
;
;    !P.CHARSIZE  = widgetData.default_charsize    
;    !P.CHARTHICK = widgetData.default_charthick
;    !P.FONT      = -1
;    !P.NOERASE   = 1
;    !P.TICKLEN   = widgetData.default_TICKLEN
;    !X.TICKS     = widgetData.default_XTICKS
;    !X.STYLE     = widgetData.default_XSTYLE
;    !Y.STYLE     = 1
;    !X.MINOR     = widgetData.default_XMINOR
;
; Overriding these values by setting keywords is of course possible
; and allowed but discouraged, since 
; then your plot might no longer have the "common" look used in
; PAPCO. The widgetData structure is PAPCO's main internal data structure
  extra_plotPar={xrange:[xut1,xut2], yrange:[yst,yen], zrange:[zmin,zmax], $
                 ylog:yscl(panel(0),3), zlog:zscl(panel(0),3), $
                 ytitle:uytitle, xtickformat:'noticks', $
                 ztitle:uztit, $
                 yticks:ytickno, yminor:4}

; check if the passed-in structure extra_par was set. If not, create
; it and set color to default black; which for the color table used by
; papco is 1. If extra_par was set, see if the tag color exists. If it
; does, leave it unchanged, if not, add it and set color to default
; black.
  if n_elements(extra_par) EQ 0 then extra_par={color:1} else begin
    names=tag_names(extra_par)
    res=where(names eq 'COLOR',count)
    if count eq 0 then extra_par=create_struct({color:1}, extra_par)
  endelse  

; add keyword structure set locally and the one passed in
  extra_plotPar=create_struct(extra_plotPar, extra_par)
   
; change the xtickformat if the panel is the bottom one and needs to
; plot a time axis   
  if panel(0) eq 0 then extra_plotPar.xtickformat='MJDT_DATE_4'
   
; You might have to add your own xtickformat for time axis
; labeling. If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label.
; You can at this point also define a custom ytickformat, or leave it out for
; IDL default plot behaviour.
   
; "down" sets the axis labeling level. 
  down=0

; do color plot and color bar plot. nodata specifies the "nodata" flag
; in your data. Here we do not plot axis and data seperate, as
; papco_plot_colorspec cannot be used as an overplot!
  
  ;check the routine papco_plot_colorspec in papco_XX\papco_lib\color_plot
  ;for all the possible options of papco_plot_colorspec.
  ;Hint: you might want to use the resx, resy keywords, which set the
  ;resolution to plot add. Without these, the routine will try to use
  ;device resolution, which for postscript may result in HUGE files!
  ;resx, resy are in data coordinates. If the axis is logarithmic, 
  ;resx, resy need to be too.
  resx=30.
  resy=32
  papco_plot_colorspec,zmat,time,yarr, nodata=nodata, resx=resx, resy=resy, $
    _extra=extra_plotPar;, /verb
  papco_color_bar,_extra=extra_plotPar
    
; store the coordinate information into common block
  plotted_x = !x
  plotted_y = !y

; plot extra x-axis labels (eg. ephemeris) and a x-axis label if required.
; plot in color=1. Example here adds an axis with date information.
; If you stick to the papco convention of always having your plot 
; time array having consecutive seconds from midnight of the start day, then 
; you can use the pre-defined xtickformat functions defined in ticks.pro in
; papco_Lib/axis_label - 0r use any other function you define yourself.
; The routine x_side_label puts x-axis label to the right of the axis.
  
  if (panel(0) eq 0) and not keyword_set(OVERPLOT) then begin
    extra_plotPar.xtickformat='MJDT_DATE_3'
    extra_plotPar.color=1
    down=1
    plot,time,time, _extra=extra_plotPar, /NODATA
    x_side_label,panel,'time (UT)!CDate'
  endif
   
; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.

  right_side_label,panel,utitle,/rot90
   
; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
  if KEYWORD_SET(OUTPUT) then begin
    print,'% p_polar_ceppad_ips_mask: Writing plot data out to file'
    description='polar_ceppad_ips_mask data'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=0
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
  endif
   
end 
