;******************************************************************************
;* PROCEDURE:     
;*      p_polar_ceppad_pitch, panel, type, OUTPUT=OUTPUT, $
;*           OVERPLOT=OVERPLOT,$
;*           PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;* 
;* DESCRIPTION:  
;*	This procedure plots data of type cep_pitch for inclusion in papco.  
;*      Data must have been read and placed into common blocks before this
;*      routine is called. Behaviour is controlled by the vector panel
;*	which controlls positioning of the plot (see papco_conventions.text
;*      in $papco). In general, the following plot conventions are used:
;*      	- if the panel is at the bottom plot time axis plus any
;*              - additional axis (ie ephemeris) needed.
;*              - if panel is at the top also plot a title
;*              - a descriptor for panels is plotted to the right of each
;*                panel, rotated by 90 deg if there is a papco_colorbar.
;*              - panels all have common time
;*              - y-scaling (or z-scaling for color plots) is either
;*              - automatic or manualy set using common yscale     
;*
;* INPUTS:       
;*	panel	three element integer vector. Controlls positioning of the plot
;*              (see papco_conventions.text in $papco)
;*      type    optional parameter for options. Normally is a four element
;*              integer vector. This is normally equivalent to the field
;*              typeVector of the papco structure papco_PLOTINFO which is set
;*              by the paneleditor. Additional fields that can be used if
;*              needed are papco_plotinfo.ioptions (20 element vector) and 
;*              papco_PLOTINFO.channel (integer). If your plot routine needs
;*              more input fileds than these you have to extend the definition
;*              of the structure papco_PLOTINFO
;*              (procedure papco_getPlotInfoStruct in $papco/papco.pro). Try to
;*              avoid this!
;*              For this data type has the following meaning:   
;*
;*                 type(0) MODE
;*                 type(1) channel number
;*                 type(2) not used
;*                 type(3) not used
;*
;*              
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;*      OVERPLOT  if you support this you can alter the appearance of your
;*              plot id it is an overplot to an exisiting panel. Standard is
;*              to plot data, no axis, using the y-scaling of the first plot
;*              and to shift the labeling on the left down one line. Useful
;*              to plot model data together with real data, ie mag. field.
;*              see p_crres_mag for an example.
;*      PLOTS_ATTHISPOSITION  used to control the lft side labeling for more
;*              than one overplot at the same position.
;*      _EXTRA  the plot structure containing some keyword for calls to
;*              plot. Used to customize color and linestyle.   
;* 
;* CALLING SEQUENCE:
;*       p_polar_ceppad_pitch, panel, type, $
;*           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,$
;*           _EXTRA=extra_par
;*
;* MODIFICATION HISTORY:       
;*     written may 1996, Reiner Friedel
;******************************************************************************
pro p_polar_ceppad_pitch, panel, plotinfo, OUTPUT=OUTPUT, $
           OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION, $
           SUBTABLE=SUBTABLE, _EXTRA=extra_par
  
forward_function polar_ceppad_pitch_comb_there
  
; It doesn't make much sense to use a color plot as an overplot, as it will
; hide what's below. Linestyles are also not useful. So the keywords 
; OVERPLOT=OVERPLOT, PLOTS_ATTHISPOSITION=PLOTS_ATTHISPOSITION,
; SUBTABLE, _EXTRA=extra_par
; are not used at all here, but are in the call for consistency...
common cep_pitch_data, input_header, input_data
common polar_ceppad_pitch_panel_data, widgetData
widgetData = plotinfo
common polar_ceppad  ;see m_polar_cep_commons.pro for contents of this block

res=polar_ceppad_pitch_comb_there() ; fill above common block
common mjdt, mjdt_start, mjdt_end    ;common time limiti in mjdt
common yscale, yscl			;man/auto yscaling
common zscale, zscl			;man/auto yscaling
common shift_label, down             ;common for x-axis label shifting
common get_error, get_err_no, get_err_msg
common coordinateSystems, plotted_x, plotted_y  
                                        ;info on coords, used by mouse
                                        ;                   functions

papco_Set_SubTable, SUBTABLE          ;set the color subtable to use
panelset,panel			;sets the panel position
   
; restrict data to actual time range requested - this makes for faster zooming
; input data is in TAI. For plotting, always start at zero to get maximum
; resolution (plot works in float only, large values get rounded!) 
tai_1=utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t*1000})
tai_2=utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t*1000})

index=where((input_data.time ge tai_1) AND (input_data.time le tai_2),c)
if c GT 2 then data=input_data(index) ELSE GOTO, noplot

xut1=0 & xut2=tai_2-tai_1 & time=data.time-tai_1

; make yarr for pitch angle bins 
pa_yarr = input_header.pa_positions 

mode = plotinfo.ioptions(2)
type = plotinfo.typeVector
ch=type(3)-1
ndat=n_elements(data)

; make yarray for channels   
yarr=fltarr(input_header.NBANDS,2)
for i=0,input_header.NBANDS-1 do begin
   yarr(i,0) = input_header.CH_POSITIONS(0,i,0,0)
   yarr(i,1) = input_header.CH_POSITIONS(1,i,0,0)
endfor  

IF pitchNames(type(0)) EQ 'HISTe' THEN BEGIN
    channels = 1
    result=polar_ceppad_flux_conv(MODE=mode, /HISTe ,CHANNELS=channels)    
    yarr = transpose(channels)
ENDIF

if type(2) eq 1 then BEGIN 
    pitch_spec='PA' 
    uytitle='Pitch Angle (deg)'
    nb=input_header.NSECTORS
    zmat=fltarr(ndat,nb)
    if type(1) eq 0 then begin ;integral channel
        intch=input_header.NBANDS-1
        low=fix(yarr(0,0)) & high=fix(yarr(input_header.NBANDS-2,1))
        ch_str='E ' + strtrim(string(low),2)  + '-' + $
                    strtrim(string(high),2) + ' keV' 
        for i=0l,ndat-1 do BEGIN
            count = data(i).count(intch,*)
            pitch = data(i).pitch(intch,*)
            index = where(count EQ 0, c)
            IF c NE 0 THEN count(index) = 1 ;don't divide by zero counts
            zmat(i,*)=pitch/count
        ENDFOR 
    endif else begin ;differential data
        ch_str='E ' + strtrim(string(fix(yarr(ch,0))),2) + '-' + $
                      strtrim(string(fix(yarr(ch,1))),2) + ' keV' 
        for i=0,ndat-1 do BEGIN
            count = data(i).count(ch,*)
            pitch = data(i).pitch(ch,*)
            index = where(count EQ 0, c)
            IF c NE 0 THEN count(index) = 1 ;don't divide by zero counts
            zmat(i,*)=pitch/count
        endfor  
    ENDELSE   
    ; make yarray for channels   
    yarr=pa_yarr
    yst=yarr(0,0) & yen=yarr(input_header.NSECTORS-1,1)
endif else BEGIN 
    ytickno=0
    pitch_spec='SP'
    uytitle='Energy (keV)'
    ch_str='PA ' + strtrim(string(fix(pa_yarr(ch,0))),2) + '!UO!N-' + $
       strtrim(string(fix(pa_yarr(ch,1))),2) + '!UO!N'
    IF pitchNames(type(0)) EQ 'IPS' THEN $
      nb=input_header.NBANDS-1 $ ;do not plot integral channel for IPS
    ELSE nb=input_header.NBANDS
    zmat=fltarr(ndat,nb)
    for i=0l,ndat-1 do begin
      count = data(i).count(0:nb-1,ch)
      pitch = data(i).pitch(0:nb-1,ch)
      index = where(count EQ 0, c)
      IF c NE 0 THEN count(index) = 1 ;don't divide by zero counts
      zmat(i,*)=pitch/count
    endfor
    yst=yarr(0,0) & yen=yarr(input_header.NBANDS-1,1)
ENDELSE 

IF pitchNames(type(0)) EQ 'HISTe' THEN BEGIN

    ;for hist, make 2D time array
    temp = fltarr(ndat, 2) & temp(*, 0) = time & temp(*, 1) = time+6.0
    time = temporary(temp)
  
    ;select data for a given mode. If first 5 channels are zero,
    ;then mode is HBC, else it's ABC
    m_arr = lonarr(ndat)
    FOR i = 0l, ndat-1 DO BEGIN
        m_arr(i) = total(data(i).count(0:5,*))
    ENDFOR
    CASE mode OF  ;zero out the OTHER mode
      0: BEGIN ;HBC mode
        index = where(m_arr GT 0, c)
        IF c NE 0 THEN FOR i = 0, c-1 DO zmat(index(i), *) = 0 
      END
      1: BEGIN ;ABC mode
        index = where(m_arr EQ 0, c) 
        IF c NE 0 THEN FOR i = 0, c-1 DO zmat(index(i), *) = 0 
      END
    ENDCASE

    ;do calibration for HISTe data
    acc=6./16. ;accumulation time for if data, 1 sector!
    channels = 1
    result=polar_ceppad_flux_conv(MODE=mode, /HISTe ,CHANNELS=channels)    
    if type(2) eq 1 then BEGIN             ;pitch angle distrib at an energy
      for i=0l,ndat-1 do zmat(i,*)=zmat(i,*)*result(ch) / acc
    ENDIF ELSE BEGIN                       ;spectrum at a pitch angle 
      for i=0l,ndat-1 do zmat(i,*)=zmat(i,*)*result / acc
    ENDELSE

    ;select "good" range of channels.
    ;zmat=zmat(*,2:14)
    ;yminarr=yminarr(2:14)  &  ymaxarr=ymaxarr(2:14)
    ;yst=yminarr(0)  &  yen=ymaxarr(n_elements(ymaxarr)-1)    

    ;Fix overlapping channels for spec only. Get midpoints, then grow
    ;channels until they join.
    nb = n_elements(yarr(*, 0))
    mid=alog10((yarr(*, 0)+yarr(*, 1))/2.0)
    dif=mid(1:n_elements(mid)-1)-mid(0:n_elements(mid)-2)
    for i=1,nb-1 do yarr(i, 0)=10^(mid(i)-dif(i-1)/2.0)
    for i=0,nb-2 do yarr(i, 1)=10^(mid(i)+dif(i)/2.0)

    input_header.cal=1
    input_header.div=1
    input_header.rat=1
ENDIF

uztit='['+input_header.ZTITLE+']'
IF pitchNames(type(0)) EQ 'HISTe' THEN BEGIN
   IF mode EQ 0 THEN mode_str = ' (HBC) ' ELSE mode_str = ' (ABC) '
ENDIF ELSE  mode_str = ' '

utitle='POLAR CEPPAD!C' + pitchNames(type(0)) + mode_str + input_header.TITLE+$
         ' '+pitch_spec+' '+strtrim(string(input_header.NSECTORS),2)
utitle=utitle+'!C'+ch_str   
  
IF input_header.rat THEN uztit = uztit + '!Ns!E-1!N'
IF input_header.cal THEN uztit = uztit + '!Ncm!E-2!Nsr!E-1!N'
IF input_header.div THEN uztit = uztit + '!NkeV!E-1!N'  
        
if (yscl(panel(0),0) eq 1) then begin
    yst=yscl(panel(0),1) & yen=yscl(panel(0),2)
endif else begin
    yscl(panel(0),1)=yst & yscl(panel(0),2)=yen
endelse

if (zscl(panel(0),0) eq 1) then begin
    zmin=zscl(panel(0),1) & zmax=zscl(panel(0),2)
endif else begin
    papco_autorange,zmat,zmin,zmax, $
      log=zscl(panel(0),3), exclude=0, nodata=0 
    zscl(panel(0),1)=zmin & zscl(panel(0),2)=zmax
endelse
      
; use auto y-labeling routine to get "good" y-lables. This sets the
; global IDL graphics system variables !Y.MINOR, !Y.TICKV, !Y.TICKS -
; so if you want to use this feature, avoid setting these or their
; corresponding keywords in your plot.
papco_y_label, yst, yen, log=yscl(panel(0),3)
     
; if this routine is called by slice to make plot arrays only
if keyword_set(OUTPUT) then if output eq 2 then return  

; set up extra plot keywords, common ones first. 
; yrange, xrange  are always needed. 
; the rest you can set if you want to override PAPCO defaults.
; See $PAPCO_BASE/PAPCO/papco_draw for the definition of the PAPCO default
; graphic sytem variable settings.
extra_plotPar={yrange:[yst, yen],xrange:[xut1, xut2], zrange:[zmin,zmax], $
                 ylog:yscl(panel(0),3), zlog:zscl(panel(0),3),  $
                 ztitle:uztit, xtickformat:'noticks', $
                 ztickformat:'papco_color_bar_log_ticks'}
  
; add keyword structure set here with the one passed in
extra_plotPar=create_struct(extra_plotPar, extra_par)
   
;do color plot and color bar plot. Use the new color plot routine from Iowa.   
;set time bin size explicitly
dx=time(1)-time(0)
resx=dx
if type(2) eq 0 then resy=(alog10(yen)-alog10(yst))/100 $ ;log scale, spectra
else                 resy=input_header.NSECTORS           ;lin scale, pitch
  
papco_plot_colorspec,zmat,time,yarr, nodata=0, resx=resx, resy=resy, $
    xstyle=5, ystyle=5, _extra=extra_plotPar

; use papco routine to draw time axis. This checks for bottom plot and uses the
; user's xtickformat if it is something other than 'noticks'.
; If you use this rotuine, make sure your data plot routine uses xstyle=5 and 
; ystyle=5 (inhibit axis) so that you don't clobber those drawn by papco.  
; "down" sets the axis labeling level. 
down=0
papco_draw_time_axis, panel, OVERPLOT=OVERPLOT, _extra=extra_plotPar

papco_color_bar,_extra=extra_plotPar
   
; plot y-axis label at left of plot. Use scalable routine!  
left_side_label,panel,uytitle,/rot90

; store the coordinate information into common block
plotted_x = !x & plotted_y = !y

; Since here papco_colorbars are plotted, labels are rotated by 90
; degrees and then plotted. No allowance is made for overplots, as
; color plots cannot be an overplot.
right_side_label,panel,utitle,/rot90

; check if data needs to be output to file. Use the optional keywords
; to add more information to the data file
;      DESCRIPTION   a string or string array describing the data
;      TIME_LABEL    a string decribing the x-axis format
;      CHANNELS      an (ch,2) array giving the channel start/end
;      Y_LABEL       a string with the y-axis label
;      Z_LABEL       a string with the z-axis label
;      RS_LABEL      a string with the righ-side label of the plot
;      NODATA        contains the no data flag value.
  
if KEYWORD_SET(OUTPUT) then begin
    message,'Writing plot data out to file'
    description='POLAR CEPPAD - spec/pitch data'
    time_label='Seconds since start of day'
    channels=yarr
    y_label=uytitle
    z_label=uztit
    rs_label=utitle
    nodata=1e-20
    papco_write_data, time, zmat, $
                      DESCRIPTION=description, TIME_LABEL=time_label, $
                      CHANNELS=channels, Y_LABEL=y_label, Z_LABEL=z_label, $
                      RS_LABEL=rs_label, NODATA=nodata
endif  
  
return

;jump to here if no data to plot 
noplot: 
  message,'No data to plot', /cont
  plotted_x = !x  &  plotted_y = !y

END 


