;******************************************************************************
;*  NAME:        r_polar_ceppad
;*
;*  DESCRIPTION: Routine to read RAL intermediate CEPPAD files. Adheres to
;*               papco conventions: Error common block, changes time format
;*               to seconds since start time.
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    INSTRUMENT   a string indicating IPS, IES or HIST
;*               MODE         a string indicating the RAL IF format
;*               LOCATION     set to LANL, MPAE, IOWA, GODDARD, AEROSPACE
;*                            to use that institution's storage conventions.
;*                            e.g. at LANL, files are kept zipped, etc
;*                            default is MPAE
;*               OLDTIME      inhibits conversion of time, used if file already
;*                            has correct format.
;*               VERBOSE: if set, print comments. Default is no comments.
;*
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     added mode KEYWORD 07.01.97 R. Friedel
;*               added  LOCATION KEYWORD 12.02.97 R. Friedel
;******************************************************************************
pro r_polar_ceppad, PlotInfo, VERBOSE=VERBOSE

COMMON polar_ceppad_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON mjdt, mjdt_start, mjdt_end              ;common time limits in mjdt
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0  
   
; clear data arrays for reuse
input_header=0  &  input_data=0

; loop through from start to end time concatonating data files into one array.
time=mjdt_start & no_files_read=0

WHILE mjdt_end.mjd GE time.mjd DO BEGIN
    utc = {MJD:long(time.mjd), TIME:long(time.t*1000)}
    datestr = ANYTIM2CAL(utc, FORM = 8, /DATE)    
    r_polar_ceppad_oneday, datestr, PlotInfo, VERBOSE=VERBOSE

    ;if a file was succesfully read, concatonate!
    if get_err_no eq 0 then begin          ;a file was read succesfully
        if no_files_read eq 0 then begin    ;start the array buffer
            input_data_buf=temporary(input_data)
        endif else begin		     ;otherwise concatonate arrays
            input_data_buf=[temporary(input_data_buf),temporary(input_data)]
      endelse 
      no_files_read=no_files_read+1
    endif
    time.mjd=time.mjd+1                   ; next day
    ; check for any time left on last day - skip if none
    if (mjdt_end.mjd eq time.mjd) AND (mjdt_end.t eq 0) then break
endwhile  
  
if (no_files_read ne 0) and (get_err_no ne 1) then begin
    get_err_no=0 & get_err_msg='cep_if read succesfully'
    input_data=0 & input_data=input_data_buf  & input_data_buf=0
endif else begin
    get_err_no=1
    return
endelse 

;time is in tai - keep it that way!
input_header.NPOINTS=n_elements(input_data)

close, /file

END 

;******************************************************************************
;*  NAME:        r_polar_ceppad_oneday
;*
;*  DESCRIPTION: Routine to read RAL intermediate CEPPAD files. Adheres to
;*               papco conventions: Error common block, changes time format
;*               to T90
;*
;*  INPUT:       datestr - string of correctly formatted day
;*
;*  KEYWORDS     INSTRUMENT - the name of the CEPPAD instrument to be
;*                            read - IES, IPS, HISTe, HISTp
;*               MODE         a string indicating the RAL IF format
;*               LOCATION     set to LANL, MPAE, IOWA, GODDARD, AEROSPACE
;*                            to use that institution's storage conventions.
;*                            e.g. at LANL, files are kept zipped, etc
;*                            default is MPAE
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        June 1996
;*
;*  AUTHOR:       R. Friedel
;*
;*  CHANGES: 
;******************************************************************************
pro r_polar_ceppad_oneday, datestr, plotinfo, VERBOSE=VERBOSE

COMMON polar_ceppad_data, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
COMMON mjdt, mjdt_start, mjdt_end           
COMMON cep_if_filename, cep_if_filename
COMMON polar_ceppad   ;see polar_ceppad_commons.pro for contents of this block

control = *plotinfo.USR_PTR1
cep_if_filename=datestr
get_err_no=0 & get_err_msg='cep_if read succesfully'

; check for env. variable pointing to data directories
if not (papco_check_data_env('PO_CEP_IF_DIR', PATH = if_path)) then return

; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0

; Read the data file. Check first if the file already exist in the correct
; format (i.e. it has already been read once and converted).
; Read file with dateformat yyyymmdd format.

;filenames are of format PO_CEP_RAL_mode_av_inst_date.dat

inst = strtrim(typeNames(control.product))
mode = varprt(control.sectors)+'Z'+varprt(control.spins)
IF mode EQ '16Z16' THEN mode = 'SURVEY' ELSE mode = 'SPIN'
IF inst EQ 'IES' THEN mode = 'CLEAN_'+mode 
product = mode+'_'+inst
yyyy = strmid(datestr, 0, 4)+'/' & yymmdd = strmid(datestr, 2, 6)

if_fln = if_path+yyyy+'PO_CEP_RAL_'+product+'_'+datestr+'.DAT'

;there are several types of data to read...
CASE 1 OF

    (control.pl_type EQ 0) OR (control.pl_type EQ 1): BEGIN
        ;detector/sector data

        file = findfile(if_fln, count = c)

        IF c EQ 0 THEN BEGIN ;make the if files from level1 files
            get_err_no = 1
            return
            message, if_fln+' not found', /info
            m_polar_ceppad, datestr, inst, mode+'_'+av
            IF get_err_no NE 0 THEN return
        ENDIF 
    END 

    (control.pl_type EQ 2) : BEGIN
        ;ips mask

    END
    
    (control.pl_type EQ 3) OR (control.pl_type EQ 4): BEGIN
        ;pitch angle files

        ;look in if directory.
        if_fln_pa = if_path+yyyy+'PO_CEP_RAL_'+product+'_PA_MF_'+datestr+'.DAT'
        file = findfile(if_fln_pa, count = c)

        IF c EQ 0 THEN BEGIN    ;make file
            ;first need to see if original if file exists.
            file = file_search(if_fln, count = c)
            IF c EQ 0 THEN BEGIN ;make if file from level1 files
                message, if_fln+' not found', /info
                m_polar_ceppad, datestr, inst, mode+'_'+av
                IF get_err_no NE 0 THEN return
            ENDIF 
            message, 'Calling ies_reprocess', /info
            ;here call the ies_reprocess wrapper to add the PA tags
            ies_reprocess, /PA, /POLAR, FILES = if_fln
            ;output is file yymmdd.dat in if directory. 
            ;Rename and move to if/yyyy/ dir
            IF NOT file_test(if_path+yymmdd+'.dat') THEN stop
            file_move, if_path+yymmdd+'.dat', if_fln_pa, /overwrite, /verb 
        ENDIF

        file = findfile(if_fln_pa, count = c)
    END 

    (control.pl_type EQ 5) : BEGIN
        ;mu range, need to read mag field data only

    END

ENDCASE 

on_ioerror, bad_restore
message, file, /info
restore, file, verbose=VERBOSE
close, /file
return
      
bad_restore:
get_err_no=2 & get_err_msg='Error opening: '+ if_fln
print, get_err_msg

END 

                              
;******************************************************************************
;*  NAME:        m_polar_ceppad
;*
;*  DESCRIPTION: Routine to make RAL intermediate CEPPAD files
;*
;*  INPUT:       datestr - string of correctly formatted day
;*
;*  KEYWORDS     none
;*
;*  OUTPUT:      none (writes files)
;*
;*  DATE:        July 2002
;*
;*  AUTHOR:      R. Friedel
;******************************************************************************
PRO m_polar_ceppad, datestr, inst, mode

COMMON get_error, get_err_no, get_err_msg

IF NOT (papco_check_data_env('PO_CEP_IF_DIR', PATH = if_path)) THEN return 

cep_make_ral_if, datestr, SINGLE = inst, MODE = mode, IF_CNTR = 2

papco_SetColorTable

;check if files where made
files = findfile(if_path+'*'+datestr+'*', count = c)

IF c EQ 0 THEN BEGIN
    get_err_no = 1
    get_err_msg = 'CEPPAD IF files could not be made'
ENDIF

get_err_no = 0

END
